<?php
//
// Nagiostats Config Wizard
// Copyright (c) 2008-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

nagiostats_configwizard_init();

function nagiostats_configwizard_init()
{
    $name = "nagiostats";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor internal performance of your XI server."),
        CONFIGWIZARD_DISPLAYTITLE => _("Nagiostats Wizard"),
        CONFIGWIZARD_FUNCTION => "nagiostats_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "nagiostats.png",
        CONFIGWIZARD_FILTER_GROUPS => array('nagios'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * Error suppressing function for printing session variables -> used for repopulating the form when going "back"
 *
 * @param $value
 * @param $type
 *
 * @return string
 */
function val($value, $type)
{
    if (!isset($_SESSION['nagiostats'])) return;

    switch ($type) {
        case 'svc':
            if (isset($_SESSION['nagiostats']['svc'][$value]) && $_SESSION['nagiostats']['svc'][$value] != '') return 'checked="checked"';
            break;

        case 'ntf':
            if (isset($_SESSION['nagiostats']['ntf'][$value]) && $_SESSION['nagiostats']['svc'][$value] != '') return 'checked="checked"';
            break;

        case 'warn':
        case 'crit':
        default:
            if (isset($_SESSION['nagiostats'][$type][$value]) && $value != '') return $_SESSION['nagiostats'][$type][$value];
    }
}

/**
 * @param string $mode
 * @param        $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function nagiostats_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "nagiostats";

    // initialize return code and output
    $result = 0;
    $output = "";

    // initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            unset($_SESSION['nagiostats']);
            $hostname = isset($_SESSION['nagiostats']['hostname']) ? $_SESSION['nagiostats']['hostname'] : 'localhost';
            $hostname = nagiosccm_replace_user_macros($hostname);

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            $hostname = grab_array_var($inargs, "hostname") == '' ? 'localhost' : encode_form_val(grab_array_var($inargs, "hostname"));
            $_SESSION['nagiostats']['hostname'] = $hostname;

            $latency = _("The amount of seconds that a scheduled check lagged behind its scheduled check time. For instance, if a check was scheduled for 03:14:15 and it didn't get executed until 03:14:17, there would be a check latency of 2.0 seconds. On-demand checks have a latency of zero seconds.");

            $buffer = _("Buffers slots are used to hold external commands that have been read from the external command file (by a separate thread) before they are processed by the Nagios daemon. If your Nagios daemon is receiving a lot of passive checks or external commands, you could end up in a situation where the buffers are always full. This results in child processes (external scripts, NSCA daemon, etc.) blocking when they attempt to write to the external command file.");

            $execTime = _("A number indicating the amount of seconds that the check took to execute (i.e. the amount of time the check was executing).");

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3OPTS:

            $outargs[CONFIGWIZARD_HIDDEN_OPTIONS] = array();

            $outargs[CONFIGWIZARD_OVERRIDE_OPTIONS] = array(
                "max_check_attempts" => 3,
            );

            $result = CONFIGWIZARD_HIDE_OPTIONS;
            $output .= "<p><strong>" . _('Note:') . " </strong>" . _('Enabling Notifications for this wizard are specified on the previous page. Click') . "
                    <strong>" . _('Next') . "</strong> " . _('to set up contacts and notification details.') . "";
            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            $services = grab_array_var($inargs, "svc", array());
            $notifications = grab_array_var($inargs, "ntf", array());
            $warnings = grab_array_var($inargs, "warn", array());
            $criticals = grab_array_var($inargs, "crit", array());

            $_SESSION['nagiostats']['svc'] = $services; //save data
            $_SESSION['nagiostats']['ntf'] = $notifications;
            $_SESSION['nagiostats']['warn'] = $warnings;
            $_SESSION['nagiostats']['crit'] = $criticals;
            //serialize and encode arrays to pass along

            $output .= '<p>' . _('Click') . ' <b>' . _('Next') . '</b> ' . _('to continue.') . '</p>';

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETSTAGE4OPTS:

            $outargs[CONFIGWIZARD_HIDDEN_OPTIONS] = array(
                CONFIGWIZARD_HIDE_NOTIFICATION_OPTIONS,
                //CONFIGWIZARD_HIDE_CHECK_INTERVAL,
                //CONFIGWIZARD_HIDE_RETRY_INTERVAL,
            );

            $output .= _("Notification options for each service are defined on the previous page....");
            $result = CONFIGWIZARD_HIDE_OPTIONS;

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:
            //pull tmp session vars
            $services = $_SESSION['nagiostats']['svc'];
            $notifications = $_SESSION['nagiostats']['ntf'];
            $warnings = $_SESSION['nagiostats']['warn'];
            $criticals = $_SESSION['nagiostats']['crit'];
            $hostname = $_SESSION['nagiostats']['hostname'];

            $objs = array();
            //change to $key value
            $i = 0;
            foreach ($services as $key => $value) {
                $notifs = 0;
                if (array_key_exists($key, $notifications) && $notifications[$key] == 'on') $notifs = 1;
                $warn = isset($warnings[$key]) ? $warnings[$key] : '';
                $crit = isset($criticals[$key]) ? $criticals[$key] : '';


                //default config properties
                $objs[$i] = array(
                    "type" => OBJECTTYPE_SERVICE,
                    "host_name" => $hostname,
                    "service_description" => '', //to be filled in below
                    "use" => "xiwizard_nagiostats_service",
                    "check_command" => "check_nagiosxi_performance!{$key}!{$warn}!{$crit}",
                    "_xiwizard" => $wizard_name,
                    "notifications_enabled" => $notifs,
                    "check_interval" => 5,
                );
                // see which services we should monitor
                switch ($key) { //service descriptions          //TODO  add check intervals
                    //active checks
                    case "ahc1":
                        $objs[$i]['service_description'] = 'ActiveHostChecks 1min';
                        $objs[$i]['check_interval'] = 1;
                        break;
                    case "ahc5":
                        $objs[$i]['service_description'] = 'ActiveHostChecks 5min';
                        break;
                    case "ahc15":
                        $objs[$i]['service_description'] = 'ActiveHostChecks 15min';
                        $objs[$i]['check_interval'] = 15;
                        break;
                    case "asc1":
                        $objs[$i]['service_description'] = 'ActiveServiceChecks 1min';
                        $objs[$i]['check_interval'] = 1;
                        break;
                    case "asc5":
                        $objs[$i]['service_description'] = 'ActiveServiceChecks 5min';
                        break;
                    case "asc15":
                        $objs[$i]['service_description'] = 'ActiveServiceChecks 15min';
                        $objs[$i]['check_interval'] = 15;
                        break;
                    //passive checks
                    case "phc1" :
                        $objs[$i]['service_description'] = 'PassiveHostChecks 1min';
                        $objs[$i]['check_interval'] = 1;
                        break;
                    case "phc5" :
                        $objs[$i]['service_description'] = 'PassiveHostChecks 5min';
                        break;
                    case "phc15" :
                        $objs[$i]['service_description'] = 'PassiveHostChecks 15min';
                        $objs[$i]['check_interval'] = 15;
                        break;
                    case "psc1" :
                        $objs[$i]['service_description'] = 'PassiveServiceChecks 1min';
                        $objs[$i]['check_interval'] = 1;
                        break;
                    case "psc5" :
                        $objs[$i]['service_description'] = 'PassiveServiceChecks 5min';
                        break;
                    case "psc15" :
                        $objs[$i]['service_description'] = 'PassiveServiceChecks 15min';
                        $objs[$i]['check_interval'] = 15;
                        break;

                    //execution and latency times
                    case "hxt":
                        $objs[$i]['service_description'] = 'AvgHostExecTime';
                        break;
                    case "sxt":
                        $objs[$i]['service_description'] = 'AvgServiceExecTime';
                        break;
                    case "ahlat":
                        $objs[$i]['service_description'] = 'AvgActiveHostLatency';
                        break;
                    case "aslat":
                        $objs[$i]['service_description'] = 'AvgActiveServiceLatency';
                        break;
                    case "phlat":
                        $objs[$i]['service_description'] = 'AvgPassiveHostLatency';
                        break;
                    case "pslat":
                        $objs[$i]['service_description'] = 'AvgPassiveHostLatency';
                        break;

                    // external command usage
                    case "eco1":
                        $objs[$i]['service_description'] = 'ExternalCommandsUsed 1min';
                        $objs[$i]['check_interval'] = 1;
                        break;
                    case "eco5":
                        $objs[$i]['service_description'] = 'ExternalCommandsUsed 5min';
                        break;
                    case "eco15":
                        $objs[$i]['service_description'] = 'ExternalCommandsUsed 1min';
                        $objs[$i]['check_interval'] = 1;
                        break;

                    // problem count
                    case "sprob":
                        $objs[$i]['service_description'] = 'Total Service Problems';
                        break;
                    case "hprob":
                        $objs[$i]['service_description'] = 'Total Host Problems';
                        break;
                    default:
                        unset($objs[$i]); // unset object if there's no service description defined
                        break;
                }

                $i++; // increment for objects

            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;
            unset($_SESSION['nagiostats']); // kill temp session data

            break;

        default:
            break;
    }

    return $output;
}