<?php
//
// Linux Server Config Wizard
// Copyright (c) 2018-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

ec2_configwizard_init();

function ec2_configwizard_init()
{
    $name = "ec2";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.1.0",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor an Amazon EC2 Instance."),
        CONFIGWIZARD_DISPLAYTITLE => "Amazon EC2",
        CONFIGWIZARD_FUNCTION => "ec2_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "ec2.png",
        CONFIGWIZARD_FILTER_GROUPS => array('amazon'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @return int
 *          0 = good to go
 *          1 = prereqs non-existent
 *          2 = may need to upgrade boto3
 */

function ec2_configwizard_check_prereqs()
{
    // Plugin doesn't exist
    if (!file_exists("/usr/local/nagios/libexec/check_ec2.py")) {
        return 4; // plugin non-existent
    }

    $preferred_version = '1.4.7';

    $found_pip = false;
    $pip_output = array();

    // See if boto3 is installed via pip and get the version number
    // $pip_command = "pip show boto3"; <-- THIS HAS LOW PERFORMANCE SOMETIMES, DO NOT USE
    $pip_command = 'python -c "import boto3"';
    exec($pip_command, $pip_output, $pip_return);

    // If neither yum nor pip returned anything, there is no need to continue
    if ($pip_return !== 0) {
        return 1; // prereqs non-existent
    }
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */

function ec2_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{

    $wizard_name = "ec2";
    $local_url = get_base_url();

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    // If this is a fresh run of the wizard, clear old data
    if (isset($_SESSION['instancelist'])) {
        unset($_SESSION['instancelist']);
    }

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $check_prereqs = ec2_configwizard_check_prereqs();

            if ($check_prereqs == 1) {
                ob_start();
?>
    <div class="alert alert-danger w-50">
        <div class="align-items-center d-flex pb-3">
            <i class="fa fa-exclamation-triangle fa-2x pe-2"></i>
            <span class="fw-bold pe-2"><?= _('Error') ?>:</span> <?= _('The python boto3 module is missing from the Nagios XI server') ?>.
        </div>
        <div class="card card-p">
            <p class="text-start">This wizard requires python module, boto3. &nbsp;Run one of the following to install boto3:</p>
            <pre><code>python -m pip install boto3</code></pre>
            <p class="text-start">OR</p>
            <pre><code>pip install boto3</code></pre>
        </div>
    </div>
<?php
                $output = ob_get_clean();

            } elseif ($check_prereqs == 4) {
                ob_start();
?>
    <div class="alert alert-danger w-50">
        <div class="align-items-center d-flex">
            <i class="fa fa-exclamation-triangle fa-2x pe-2"></i>
            <span class="fw-bold pe-2">Error:</span> <?= _('The check_ec2.py plugin is missing from the Nagios XI server') ?>.
        </div>
    </div>
<?php
                $output = ob_get_clean();

            } else {
                $instanceid = grab_array_var($inargs, "instanceid", "");
                $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
                $accesskey = grab_array_var($inargs, "accesskey", "");
                $staticcreds = grab_array_var($inargs, "staticcreds", "");
                $staticconfig = grab_array_var($inargs, "staticconfig", "");
                $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
                $configfilepath = grab_array_var($inargs, "configfilepath", "");
                $region = grab_array_var($inargs, "aws_region", "");
                $instance_list = grab_array_var($inargs, 'instancelist', '');

                if (gettype($instance_list) == 'string') {
                    $instance_list = json_decode(base64_decode($instance_list), true);
                }

                if ($credsfilepath == "") {
                    $credsfilepath = "/usr/local/nagiosxi/etc/.aws/credentials";
                }

                if ($configfilepath == "") {
                    $configfilepath = "/usr/local/nagiosxi/etc/.aws/config";
                }

                $linuxdistro = grab_array_var($inargs, "linuxdistro", "");

                # Get the existing host/node configurations.
                # TODO: Include passwords/secrets?
                $nodes = get_configwizard_hosts($wizard_name);

                ########################################################################################
                # Load the html
                # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
                #   to load the PHP from the Step1 file into the $output string.
                ########################################################################################
                ob_start();
                include __DIR__.'/steps/step1.php';
                $output = ob_get_clean();
            }

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");

            $staticcreds = grab_array_var($inargs, "staticcreds", "off");
            $staticconfig = grab_array_var($inargs, "staticconfig", "off");

            $checkbox_list = grab_array_var($inargs, "checkboxlist", "");
            $instance_list = grab_array_var($inargs, "instancelist", "");

            if (isset($checkbox_list)) {
                $_SESSION['checkbox_list'] = $checkbox_list;
            }

            if (isset($instance_list)) {
                $_SESSION['instance_list'] = $instance_list;
            }

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (empty($accesskeyid) and $staticcreds == "off") {
                $errmsg[$errors++] = _("Specify either an Access Key ID or a static credentials file.");
            }

            if (empty($accesskey) and $staticcreds == "off") {
                $errmsg[$errors++] = _("Specify either a Secret Access Key or a static credentials file.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");

            $instance_list = grab_array_var($inargs, "instancelist", "");

            if (gettype($instance_list) == 'string') {
                $instance_list = json_decode(base64_decode($instance_list), true);
            }

            $hostname = grab_array_var($inargs, "hostname");
            /*<input name="instancelist[' . $instance_id . '][hostname]" value="' . $instance["ip_address"] . '" class="textarea form-control" type="text"/>*/
            $password = "";
            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services_default = array(
                    "ping" => 1,
                    "CPUCreditUsage" => 1,
                    "CPUCreditBalance" => 1,
                    "CPUUtilization" => 1,
                    "DiskReadOps" => 1,
                    "DiskWriteOps" => 1,
                    "DiskReadBytes" => 1,
                    "DiskWriteBytes" => 1,
                    "NetworkIn" => 1,
                    "NetworkOut" => 1,
                    "NetworkPacketsIn" => 1,
                    "NetworkPacketsOut" => 1,
                    "StatusCheckFailed" => 1,
                    "StatusCheckFailed_Instance" => 1,
                    "StatusCheckFailed_System" => 1

                );

                $services_default["servicestate"][0] = "on";
                $services_default["servicestate"][1] = "on";
                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (!is_array($serviceargs)) {
                $serviceargs_default = array(

                    "CPUCreditUsage_warning" => 60,
                    "CPUCreditUsage_critical" => 120,

                    "CPUCreditBalance_warning" => 100,
                    "CPUCreditBalance_critical" => 25,

                    "CPUUtilization_warning" => 50,
                    "CPUUtilization_critical" => 75,

                    "DiskReadOps_warning" => 15000,
                    "DiskReadOps_critical" => 30000,

                    "DiskWriteOps_warning" => 15000,
                    "DiskWriteOps_critical" => 30000,

                    "DiskReadBytes_warning" => 1000000000,
                    "DiskReadBytes_critical" => 2000000000,

                    "DiskWriteBytes_warning" => 1000000000,
                    "DiskWriteBytes_critical" => 2000000000,

                    "NetworkIn_warning" => 1000000000,
                    "NetworkIn_critical" => 2000000000,

                    "NetworkOut_warning" => 1000000000,
                    "NetworkOut_critical" => 2000000000,

                    "NetworkPacketsIn_warning" => 1750000,
                    "NetworkPacketsIn_critical" => 3500000,

                    "NetworkPacketsOut_warning" => 1750000,
                    "NetworkPacketsOut_critical" => 3500000,

                    "StatusCheckFailed_warning" => 1,
                    "StatusCheckFailed_critical" => 1,

                    "StatusCheckFailed_Instance_warning" => 1,
                    "StatusCheckFailed_Instance_critical" => 1,

                    "StatusCheckFailed_System_warning" => 1,
                    "StatusCheckFailed_System_critical" => 1,
                );

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            $main_css_url = get_base_url() . 'includes/configwizards/amazon_ec2/css/main'.(get_theme() === 'xi5dark' ? '-dark' : '') . '.css';

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $instanceid = grab_array_var($inargs, "instanceid", "");
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");

            $instance_list = grab_array_var($inargs, "instancelist", "");

            if (gettype($instance_list) == 'string') {
                $instance_list = json_decode(base64_decode($instance_list), true);
            }

            // Check for errors
            $errors = 0;
            $errmsg = array();
            foreach($instance_list as $instance_id => $instance) {
                if (is_valid_host_name($instance["hostname"]) == false) {
                    $errmsg[$errors++] = sprintf(_("Invalid host name for instance %s."), $instance_id);
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $instanceid = grab_array_var($inargs, "instanceid", "");
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");
            $region = grab_array_var($inargs, "aws_region", "");

            $instance_list = grab_array_var($inargs, "instancelist", "");

            if (gettype($instance_list) == 'string') {
                $instance_list = json_decode(base64_decode($instance_list), true);
            }

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial");
            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            } else {
                $services = grab_array_var($inargs, "services");
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");
            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            } else {
                $serviceargs = grab_array_var($inargs, "serviceargs");
            }

            $output = '
            <input type="hidden" name="instancelist" value="' . base64_encode(json_encode($instance_list)) . '">
            <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
            <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
            <input type="hidden" name="instanceid" value="' . encode_form_val($instanceid) . '">
            <input type="hidden" name="accesskeyid" value="' . encode_form_val($accesskeyid) . '">
            <input type="hidden" name="accesskey" value="' . encode_form_val($accesskey) . '">
            <input type="hidden" name="aws_region" value="' . encode_form_val($region) . '">
            <input type="hidden" name="staticcreds" value="' . encode_form_val($staticcreds) . '">
            <input type="hidden" name="staticconfig" value="' . encode_form_val($staticconfig) . '">
            <input type="hidden" name="credsfilepath" value="' . encode_form_val($credsfilepath) . '">
            <input type="hidden" name="configfilepath" value="' . encode_form_val($configfilepath) . '">
            <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
            <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            $check_interval = grab_array_var($inargs, "check_interval", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();
            if ($check_interval < 5) {
                $errmsg[$errors++] = _("Check interval cannot be less than 5 minutes.");
            }
            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $hostname = grab_array_var($inargs, "hostname", "");
            $instanceid = grab_array_var($inargs, "instanceid", "");
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");
            $check_interval = grab_array_var($inargs, "check_interval", "");

            $instance_list = json_decode(base64_decode(grab_array_var($inargs, "instancelist", "")), true);

            $objs = array();

            foreach ($instance_list as $instance_id => $instance) {

                if (!host_exists($instance["hostname"])) {
                    $objs[] = array(
                        "type" => OBJECTTYPE_HOST,
                        "use" => "xiwizard_linuxserver_host",
                        "host_name" => $instance["hostname"],
                        "address" => $instance["ip_address"],
                        "icon_image" => "ec2.png",
                        "statusmap_image" => "ec2.png",
                        "_xiwizard" => $wizard_name,
                    );
                }

                $commonopts = "--instanceid '" . $instance["instance_id"] . "' ";

                // Append credentials file path if toggled, otherwise pass in instanceid, accesskeyid and accesskey
                if ($staticcreds == "on") {
                    $commonopts .= "--credfile '" . $credsfilepath . "' ";
                } else {
                    $commonopts .= "--accesskeyid '" . $accesskeyid . "' --secretaccesskey '" . $accesskey . "' ";
                }
                // Append config file path if toggled, otherwise pass in region
                if ($staticconfig == "on") {
                    $commonopts .= "--configfile '" . $configfilepath . "' ";
                } else {
                    $commonopts .= "--region '" . $instance["region"] . "' ";
                }

                $metric_name_dict = array(

                        "CPUCreditBalance" => "CPU Credit Balance",
                        "CPUCreditUsage" => "CPU Credit Usage",
                        "CPUUtilization" => "CPU Utilization",
                        "DiskReadBytes" => "Disk Read Bytes",
                        "DiskReadOps" => "Disk Read Ops",
                        "DiskWriteBytes" => "Disk Write Bytes",
                        "DiskWriteOps" => "Disk Write Ops",
                        "NetworkIn" => "Network In",
                        "NetworkOut" => "Network Out",
                        "NetworkPacketsIn" => "Network Packets In",
                        "NetworkPacketsOut" => "Network Packets Out",
                        "StatusCheckFailed" => "Status Check Failed",
                        "StatusCheckFailed_Instance" => "Status Check Failed Instance",
                        "StatusCheckFailed_System" => "Status Check Failed System"

                );

                foreach ($instance["metrics"] as $metric_name => $metric_info) {
                    if ( $metric_info["checkbox"] != "on") {
                        continue;
                    }

                    switch ($metric_name) {

                        case "ping":
                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $instance["hostname"],
                                "service_description" => "Ping",
                                "use" => "xiwizard_linuxserver_ping_service",
                                "_xiwizard" => $wizard_name,
                            );
                            break;

                        default:
                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $instance["hostname"],
                                "service_description" => $metric_name_dict[$metric_name],
                                "use" => "xiwizard_linuxserver_ping_service",
                                "check_command" => "check_ec2!" . " -P " . $check_interval . " --metricname " . $metric_name . " " . $commonopts . "--warning '" . $metric_info["warning"] . "' --critical '" . $metric_info["critical"] . "'",
                                "_xiwizard" => $wizard_name,
                            );
                            break;
                    }
                }
            }

            // Return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


/**
 *
 * @return string
 */
