<?php
//
// Linux Server Config Wizard
// Copyright (c) 2018-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

s3_configwizard_init();

function s3_configwizard_init()
{
    $name = "s3";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor an Amazon S3 Bucket."),
        CONFIGWIZARD_DISPLAYTITLE => "Amazon S3",
        CONFIGWIZARD_FUNCTION => "s3_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "s3.png",
        CONFIGWIZARD_FILTER_GROUPS => array('amazon'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100,
    );
    register_configwizard($name, $args);
}

/**
 * @return int
 *          0 = good to go
 *          1 = prereqs non-existent
 *          2 = may need to upgrade boto3
 */

function s3_configwizard_check_prereqs()
{
    // Plugin doesn't exist
    if (!file_exists("/usr/local/nagios/libexec/check_s3.py") && !is_dev_mode()) {
        return 4; // plugin non-existent
    }

    $preferred_version = '1.4.7';

    $found_pip = false;
    $pip_output = array();

    // See if boto3 is installed via pip and get the version number
    // $pip_command = "pip show boto3"; <-- THIS HAS LOW PERFORMANCE SOMETIMES, DO NOT USE
    $pip_command = 'python -c "import boto3"';
    exec($pip_command, $pip_output, $pip_return);

    // If neither yum nor pip returned anything, there is no need to continue
    if ($pip_return !== 0) {
        return 1; // prereqs non-existent
    }
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */

function s3_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "s3";
    $local_url = get_base_url();

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $check_prereqs = s3_configwizard_check_prereqs();

            // Remove any existing session variables if this is the first time on the page
            if (isset($_POST['backButton']) == false) {
                unset($_SESSION['bucketlist']);
                unset($_SESSION['checkboxlist']);
            }

            if ($check_prereqs == 1) {
                ob_start();
?>
    <div class="alert alert-danger w-50">
        <div class="align-items-center d-flex pb-3">
            <i class="fa fa-exclamation-triangle fa-2x pe-2"></i>
            <span class="fw-bold pe-2"><?= _('Error') ?>:</span> <?= _('The python boto3 module is missing from the Nagios XI server') ?>.
        </div>
        <div class="card card-p">
            <p class="text-start">This wizard requires python module, boto3. &nbsp;Run one of the following to install boto3:</p>
            <pre><code>python -m pip install boto3</code></pre>
            <p class="text-start">OR</p>
            <pre><code>pip install boto3</code></pre>
            <p class="text-start"><?= _('CentOS/RHEL 7 may require python-futures, to display buckets.') ?></p>
            <pre><code>yum install python-futures</code></pre>
        </div>
    </div>
<?php
                $output = ob_get_clean();

            } elseif ($check_prereqs == 4) {
                ob_start();
?>
    <div class="alert alert-danger w-50">
        <div class="align-items-center d-flex">
            <i class="fa fa-exclamation-triangle fa-2x pe-2"></i>
            <span class="fw-bold pe-2">Error:</span> <?= _('The check_ec2.py plugin is missing from the Nagios XI server') ?>.
        </div>
    </div>
<?php
                $output = ob_get_clean();

            } else {

                // Since this is the first step, we always want to reference the bucketlist stored in session
                $bucket_list = (array_key_exists('bucketlist', $_SESSION) ? $_SESSION['bucketlist'] : '');

                $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
                $accesskey = grab_array_var($inargs, "accesskey", "");
                $staticcreds = grab_array_var($inargs, "staticcreds", "");
                $staticconfig = grab_array_var($inargs, "staticconfig", "");
                $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
                $configfilepath = grab_array_var($inargs, "configfilepath", "");

                if ($credsfilepath == "") {
                    $credsfilepath = "/usr/local/nagiosxi/etc/.aws/credentials";
                }

                if ($configfilepath == "") {
                    $configfilepath = "/usr/local/nagiosxi/etc/.aws/config";
                }

                # Get the existing host/node configurations.
                # TODO: Include passwords/secrets?
                $nodes = get_configwizard_hosts($wizard_name);

                ########################################################################################
                # Load the html
                # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
                #   to load the PHP from the Step1 file into the $output string.
                ########################################################################################
                ob_start();
                include __DIR__.'/steps/step1.php';
                $output = ob_get_clean();
            }

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            $bucket_list = grab_array_var($inargs, "bucketlist", "");

            // If we for some reason don't have a bucketlist, grab the session
            // Traversing 'Back' from Step 3, bucketlist will be json_encoded, so just grab the session.
            if ($bucket_list == "" || isset($_POST['backButton']) == true) {
                $bucket_list = (array_key_exists('bucketlist', $_SESSION) ? $_SESSION['bucketlist'] : "");
            }

            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");
            $regionlist = grab_array_var($inargs, "regionlist", "");

            $staticcreds = grab_array_var($inargs, "staticcreds", "off");
            $staticconfig = grab_array_var($inargs, "staticconfig", "off");

            // Check for errors
            $errors = 0;
            $errmsg = array();
            $number_of_checked_buckets = 0;

            if (is_array($bucket_list)) {
                foreach ($bucket_list as $bucket_name => $bucket_info) {
                    if ($bucket_info['checkbox'] == 'on') {
                        $number_of_checked_buckets++;
                    }
                }
            }

            if ($number_of_checked_buckets == 0) {
                $errmsg[$errors++] = _("You must select at least one bucket to continue");
            }

            if (empty($accesskeyid) and $staticcreds == "off") {
                $errmsg[$errors++] = _("Specify either an Access Key ID or a static credentials file");
            }

            if (empty($accesskey) and $staticcreds == "off") {
                $errmsg[$errors++] = _("Specify either a Secret Access Key or a static credentials file");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            // Take each bucket name and get available metrics, store them in an array
            if (is_array($bucket_list)) {
                foreach($bucket_list as $bucket_name => $bucket_info) {

                    if (!$bucket_info['checkbox'] ) {
                        // If the checkbox isn't on, skip it
                        continue;

                    } else {

                        // Don't overwrite if we've been here before
                        if ( isset($bucket_info['metrics']) && !empty($bucket_info['metrics']) && $bucket_info['metrics'] != 'Array') {
                            continue;
                        }

                        // If static files are specified, use those
                        if ( $staticcreds !== 'off' && !empty($credsfilepath)) {
                            $cmd = "/usr/local/nagios/libexec/check_s3.py --changemode 'getmetrics' --credfile '" . $credsfilepath . "' --bucketname '" . $bucket_name . "' --region '" . $bucket_info['region'] . "'";
                        } else {
                            // No static file defined
                            $cmd = "/usr/local/nagios/libexec/check_s3.py --changemode 'getmetrics' --accesskeyid '" . $accesskeyid . "' --secretaccesskey '" . $accesskey . "' --bucketname '" . $bucket_name . "' --region '" . $bucket_info['region'] . "'";
                        }

                        $cmd = escapeshellcmd($cmd);

                        $returnedmetrics = shell_exec($cmd);
                        $decodedmetrics = json_decode($returnedmetrics, true);

                        $merged_array = array();

                        foreach($decodedmetrics as $index => $metric) {
                            $merged_array[$index] = $metric;
                        }

                        $bucket_list[$bucket_name]['metrics'] = base64_encode(json_encode($merged_array));
                    }
                }
            }

            // Cache the bucketlist again after adding metrics
            $_SESSION['bucketlist'] = $bucket_list;

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us

            // $bucket_list = grab_array_var($inargs, "bucketlist", "");
            $bucket_list = $_SESSION['bucketlist'];
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");
            $regionlist = grab_array_var($inargs, "regionlist", "");
            $checkboxlist = grab_array_var($inargs, "checkboxlist", "");

            // Use to grab necessary information by condensed metric name
            $metrics_dictionary = array(
                'BucketSizeBytes' => array(
                    'name' => 'Bucket Size Bytes',
                    'description' => 'The amount of data in bytes stored in a bucket in the STANDARD storage class, INTELLIGENT_TIERING storage class, Standard - Infrequent Access (STANDARD_IA) storage class, OneZone - Infrequent Access (ONEZONE_IA), Reduced Redundancy Storage (RRS) class, or Glacier (GLACIER) storage class',
                    'units' => 'Bytes'
                ),
                'NumberOfObjects' => array(
                    'name' => 'Number of Objects',
                    'description' => 'The total number of objects (includes all objects including all their versions) stored in a bucket for all storage classes',
                    'units' => 'Objects'
                ),
                'AllRequests' => array(
                    'name' => 'All Requests',
                    'description' => 'The total number of HTTP requests made to an Amazon S3 bucket, regardless of type. If you\'re using a metrics configuration with a filter, then this metric only returns the HTTP requests made to the objects in the bucket that meet the filter\'s requirements',
                    'units' => 'Requests'
                ),
                'GetRequests' => array(
                    'name' => 'Get Requests',
                    'description' => 'The number of HTTP GET requests made for objects in an Amazon S3 bucket. This doesn\'t include list operations',
                    'units' => 'Requests'
                ),
                'PutRequests' => array(
                    'name' => 'Put Requests',
                    'description' => 'The number of HTTP PUT requests made for objects in an Amazon S3 bucket',
                    'units' => 'Requests'
                ),
                'DeleteRequests' => array(
                    'name' => 'Delete Requests',
                    'description' => 'The number of HTTP DELETE requests made for objects in an Amazon S3 bucket. This also includes Delete Multiple Objects requests. This metric shows the number of requests, not the number of objects deleted',
                    'units' => 'Requests'
                ),
                'HeadRequests' => array(
                    'name' => 'Head Requests',
                    'description' => 'The number of HTTP HEAD requests made to an Amazon S3 bucket',
                    'units' => 'Requests'
                ),
                'PostRequests' => array(
                    'name' => 'Post Requests',
                    'description' => 'The number of HTTP POST requests made to an Amazon S3 bucket',
                    'units' => 'Requests'
                ),
                'ListRequests' => array(
                    'name' => 'List Requests',
                    'description' => 'The number of HTTP requests that list the contents of a bucket',
                    'units' => 'Requests'
                ),
                'BytesDownloaded' => array(
                    'name' => 'Bytes Downloaded',
                    'description' => 'The number bytes downloaded for requests made to an Amazon S3 bucket, where the response includes a body',
                    'units' => 'Bytes'
                ),
                'BytesUploaded' => array(
                    'name' => 'Bytes Uploaded',
                    'description' => 'The number bytes uploaded that contain a request body, made to an Amazon S3 bucket',
                    'units' => 'Bytes'
                ),
                '4xxErrors' => array(
                    'name' => '4xx Errors',
                    'description' => 'The number of HTTP 4xx client error status code requests made to an Amazon S3 bucket with a value of either 0 or 1. The average statistic shows the error rate, and the sum statistic shows the count of that type of error, during each period',
                    'units' => 'Errors'
                ),
                '5xxErrors' => array(
                    'name' => '5xx Errors',
                    'description' => 'The number of HTTP 5xx server error status code requests made to an Amazon S3 bucket with a value of either 0 or 1. The average statistic shows the error rate, and the sum statistic shows the count of that type of error, during each period',
                    'units' => 'Errors'
                ),
                'FirstByteLatency' => array(
                    'name' => 'First Byte Latency',
                    'description' => 'The per-request time from the complete request being received by an Amazon S3 bucket to when the response starts to be returned',
                    'units' => 'Milliseconds'
                ),
                'TotalRequestLatency' => array(
                    'name' => 'Total Request Latency',
                    'description' => 'The elapsed per-request time from the first byte received to the last byte sent to an Amazon S3 bucket. This includes the time taken to receive the request body and send the response body, which is not included in First Byte Latency',
                    'units' => 'Milliseconds'
                ),
            );

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services_default = array(
                    "ping" => 1,
                    "BucketSizeBytes" => 1,
                    "NumberOfObjects" => 1,
                    "AllRequests" => 1,
                    "GetRequests" => 1,
                    "PutRequests" => 1,
                    "DeleteRequests" => 1,
                    "HeadRequests" => 1,
                    "PostRequests" => 1,
                    "ListRequests" => 1,
                    "BytesDownloaded" => 1,
                    "BytesUploaded" => 1,
                    "4xxErrors" => 1,
                    "5xxErrors" => 1,
                    "FirstByteLatency" => 1,
                    "TotalRequestLatency" => 1

                );

                $services_default["servicestate"][0] = "on";
                $services_default["servicestate"][1] = "on";
                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if ($serviceargs_serial != "") {
                $serviceargs_default = json_decode(base64_decode($serviceargs_serial), true);
            } else {
                $serviceargs_default = array(

                    "BucketSizeBytes_warning" => 1000000000,
                    "BucketSizeBytes_critical" => 1500000000,

                    "NumberOfObjects_warning" => 150,
                    "NumberOfObjects_critical" => 200,

                    "AllRequests_warning" => 1000,
                    "AllRequests_critical" => 1500,

                    "GetRequests_warning" => 1000,
                    "GetRequests_critical" => 1500,

                    "PutRequests_warning" => 1000,
                    "PutRequests_critical" => 1500,

                    "DeleteRequests_warning" => 1000,
                    "DeleteRequests_critical" => 1500,

                    "HeadRequests_warning" => 1000,
                    "HeadRequests_critical" => 1500,

                    "PostRequests_warning" => 1000,
                    "PostRequests_critical" => 1500,

                    "ListRequests_warning" => 1000,
                    "ListRequests_critical" => 1500,

                    "BytesDownloaded_warning" => 500000000,
                    "BytesDownloaded_critical" => 750000000,

                    "BytesUploaded_warning" => 500000000,
                    "BytesUploaded_critical" => 750000000,

                    "4xxErrors_warning" => 25,
                    "4xxErrors_critical" => 50,

                    "5xxErrors_warning" => 25,
                    "5xxErrors_critical" => 50,

                    "FirstByteLatency_warning" => 150,
                    "FirstByteLatency_critical" => 300,

                    "TotalRequestLatency_warning" => 150,
                    "TotalRequestLatency_critical" => 300
                );

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            if (!is_array($serviceargs)) {
                $serviceargs_default = array(

                    "BucketSizeBytes_warning" => 1000000000,
                    "BucketSizeBytes_critical" => 1500000000,

                    "NumberOfObjects_warning" => 150,
                    "NumberOfObjects_critical" => 200,

                    "AllRequests_warning" => 1000,
                    "AllRequests_critical" => 1500,

                    "GetRequests_warning" => 1000,
                    "GetRequests_critical" => 1500,

                    "PutRequests_warning" => 1000,
                    "PutRequests_critical" => 1500,

                    "DeleteRequests_warning" => 1000,
                    "DeleteRequests_critical" => 1500,

                    "HeadRequests_warning" => 1000,
                    "HeadRequests_critical" => 1500,

                    "PostRequests_warning" => 1000,
                    "PostRequests_critical" => 1500,

                    "ListRequests_warning" => 1000,
                    "ListRequests_critical" => 1500,

                    "BytesDownloaded_warning" => 500000000,
                    "BytesDownloaded_critical" => 750000000,

                    "BytesUploaded_warning" => 500000000,
                    "BytesUploaded_critical" => 750000000,

                    "4xxErrors_warning" => 25,
                    "4xxErrors_critical" => 50,

                    "5xxErrors_warning" => 25,
                    "5xxErrors_critical" => 50,

                    "FirstByteLatency_warning" => 150,
                    "FirstByteLatency_critical" => 300,

                    "TotalRequestLatency_warning" => 150,
                    "TotalRequestLatency_critical" => 300
                );

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            $main_css_url = get_base_url() . 'includes/configwizards/amazon_s3/css/main'.(get_theme() === 'xi5dark' ? '-dark' : '') . '.css';

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $bucket_list = grab_array_var($inargs, "bucketlist", "");
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");
            $region = grab_array_var($inargs, "region", "");

            // Cache our bucketlist
            $_SESSION['bucketlist'] = $bucket_list;

            // Check for errors
            $errors = 0;
            $errmsg = array();

            foreach ($hostnames as $host) {

                if (is_valid_host_name($host) == false) {
                    $errmsg[$errors++] = "[" . $host . "] Is an invalid host name.";
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $hostnames = grab_array_var($inargs, "hostnames");
            $bucket_list = grab_array_var($inargs, "bucketlist", "");
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");

            $output = '
            <input type="hidden" name="bucketlist" value="' . base64_encode(json_encode($bucket_list)) . '">
            <input type="hidden" name="accesskeyid" value="' . encode_form_val($accesskeyid) . '">
            <input type="hidden" name="accesskey" value="' . encode_form_val($accesskey) . '">
            <input type="hidden" name="region" value="' . encode_form_val($region) . '">
            <input type="hidden" name="staticcreds" value="' . encode_form_val($staticcreds) . '">
            <input type="hidden" name="staticconfig" value="' . encode_form_val($staticconfig) . '">
            <input type="hidden" name="credsfilepath" value="' . encode_form_val($credsfilepath) . '">
            <input type="hidden" name="configfilepath" value="' . encode_form_val($configfilepath) . '">';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            $check_interval = grab_array_var($inargs, "check_interval", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();
            if ($check_interval < 5) {
                $errmsg[$errors++] = "Check interval cannot be less than five minutes. This is because AWS sends CloudWatch data every five minutes. Querying between the time the last data was was received and now - if less than five minutes - will result in an empty response from CloudWatch.";
            }
            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            // Get variables that were passed to us
            $bucket_list = $_SESSION['bucketlist'];
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");
            $check_interval = grab_array_var($inargs, "check_interval", "");

            $objs = array();

            // Append credentials file path if toggled, otherwise pass in bucketlist, accesskeyid and accesskey
            if ($staticcreds == "on") {
                $commonopts .= "--credfile '" . $credsfilepath . "' ";
            } else {
                $commonopts .= "--accesskeyid '" . $accesskeyid . "' --secretaccesskey '" . $accesskey . "' ";
            }

            // Append config file path if toggled, otherwise pass in region
            if ($staticconfig == "on") {
                $commonopts .= "--configfile '" . $configfilepath . "' ";
            }

            // Create non-existent hosts
            foreach ($bucket_list as $bucket_name => $bucket_info) {

                if ($bucket_info['checkbox'] != 'on') {
                    continue;
                }

                if (!host_exists($bucket_info["hostname"])) {
                    $objs[] = array(
                        "type" => OBJECTTYPE_HOST,
                        "use" => "xiwizard_linuxserver_host",
                        "host_name" => $bucket_info['hostname'],
                        "address" => $bucket_info['hostname'],
                        "icon_image" => "s3.png",
                        "statusmap_image" => "s3.png",
                        "_xiwizard" => $wizard_name,
                        "check_command" => "check_s3!" . "--changemode 'checkalive' --bucketname '" . $bucket_name . "' " . $commonopts
                    );
                }

            }

            $metric_name_dictionary = array(

                    "NumberOfObjects" => "Number of Objects",
                    "BucketSizeBytes" => "Bucket Size Bytes",
                    "AllRequests" => "All Requests",
                    "DeleteRequests" => "Delete Requests",
                    "GetRequests" => "Get Requests",
                    "HeadRequests" => "Head Requests",
                    "PutRequests" => "Put Requests",
                    "PostRequests" => "Post Requests",
                    "ListRequests" => "List Requests",
                    "BytesDownloaded" => "Bytes Downloaded",
                    "BytesUploaded" => "Bytes Uploaded",
                    "4xxErrors" => "4xx Errors",
                    "5xxErrors" => "5xx Errors",
                    "FirstByteLatency" => "First Byte Latency",
                    "TotalRequestLatency" => "Total Request Latency"

            );

            // See which services we should monitor
            foreach ($bucket_list as $bucket_name => $bucket_info) {

                $metrics = $bucket_info['metrics'];

                foreach ($metrics as $metric_name => $metric_info) {

                    if ($metric_info['checkbox'] != 'on') {
                        continue;
                    }

                    $dimensionopts = "";

                    if ($metric_info['storage_type'] != "") {
                        $dimensionopts .= "--storagetype '" . $metric_info['storage_type'] . "' ";
                    }

                    if ($metric_info['filter_id'] != "") {
                        $dimensionopts .= "--filterid '" . $metric_info['filter_id'] . "' ";
                    }

                    if ($bucket_name != "") {
                        $dimensionopts .= "--bucketname '" . $bucket_name . "' ";
                    }

                    if ($bucket_info['region'] != "" && $statticconfig != "on") {
                        $dimensionopts .= "--region '" . $bucket_info['region'] . "' ";
                    }

                    switch ($metric_name) {

                        // Set check period of one day's worth of minutes for daily metrics
                        case "BucketSizeBytes":
                        case "NumberOfObjects":

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $bucket_info['hostname'],
                                "service_description" => $bucket_info['hostname'] . " " . $metric_name_dictionary[$metric_name],
                                "use" => "xiwizard_linuxserver_ping_service",
                                "check_command" => "check_s3!" . "-P 1440" . " --metricname '" . $metric_name . "' " . $commonopts . "--warning '" . $metric_info['warning'] . "' --critical '" . $metric_info['critical'] . "' " . $dimensionopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;

                        case "Region":
                            break;

                        case "Hostname":
                            break;

                        default:
                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $bucket_info['hostname'],
                                "service_description" => $bucket_info['hostname'] . " " . $metric_name_dictionary[$metric_name],
                                "use" => "xiwizard_linuxserver_ping_service",
                                "check_command" => "check_s3!" . "-P " . $check_interval . " --metricname '" . $metric_name . "' " . $commonopts . "--warning '" . $metric_info['warning'] . "' --critical '" . $metric_info['critical'] . "' " . $dimensionopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;
                    }
                }
            }

            // Return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


/**
 *
 * @return string
 */
