<?php
//
// Auto-Discovery Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

autodiscovery_configwizard_init();

function autodiscovery_configwizard_init()
{
    $name = "autodiscovery";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _('Monitor servers, devices, and services found by auto-discovery jobs.'),
        CONFIGWIZARD_DISPLAYTITLE => _('Auto-Discovery'),
        CONFIGWIZARD_FUNCTION => 'autodiscovery_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => 'autodiscovery.png',
        CONFIGWIZARD_FILTER_GROUPS => array('nagios', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function autodiscovery_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "autodiscovery";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $job = grab_array_var($inargs, "job", "");
            $show = grab_array_var($inargs, "show", "");
            $addresstype = grab_array_var($inargs, "addresstype", "ip");
            $defaultservices = grab_array_var($inargs, "defaultservices", "common");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            $back = array_key_exists("backButton", $_POST);
            if ($back) break;
            
            // Get variables that were passed to us
            $job = grab_array_var($inargs, "job", "");
            $defaultservices = grab_array_var($inargs, "defaultservices", "common");


            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($job) == false) {
                $errmsg[$errors++] = _("No job specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $job = grab_array_var($inargs, "job");
            $show = grab_array_var($inargs, "show", "");
            $addresstype = grab_array_var($inargs, "addresstype", "ip");
            $defaultservices = grab_array_var($inargs, "defaultservices", "common");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $services = json_decode(base64_decode($services_serial), true);

            if (!is_array($services)) {
                $services = grab_array_var($inargs, "services", array());
            }

            $services_serial = base64_encode(json_encode($services));

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            $back = array_key_exists("backButton", $_POST);
            if ($back) break;
            
            // get variables that were passed to us
            $job = grab_array_var($inargs, "job");
            $show = grab_array_var($inargs, "show", "");
            $addresstype = grab_array_var($inargs, "addresstype", "ip");
            $defaultservices = grab_array_var($inargs, "defaultservices", "common");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $services_u = json_decode(base64_decode($services_serial), true);
            $services_r = grab_array_var($inargs, "services", array());

            $services = array();
            if (is_array($services_u) && is_array($services_r))
                $services = array_merge($services_u, $services_r);
            
            // check for errors
            $errors = 0;
            $errmsg = array();
            $havehost = false;
            /*
            if (is_valid_host_name($hostname)==false)
                $errmsg[$errors++]="Invalid host name.";
            */
            if (!is_array($services))
                $errmsg[$errors++] = _("Service array is empty.");
            else {
                $havesvc = false;
                $hosthost = false;

                //print_r($services);

                foreach ($services as $address => $harr) {

                    $checkthishost = false;

                    if (array_key_exists("selected", $harr)) {
                        $havesvc = true;
                        $havehost = true;
                        $checkthishost = true;
                    }

                    // validate host name
                    if ($checkthishost == true) {
                        $hostname = $harr["hostname"];
                        if (is_valid_host_name($hostname) == false)
                            $errmsg[$errors++] = "Invalid host name '<b>$hostname</b>'";
                    }

                    // check all of the host's services
                    if (array_key_exists("ports", $harr)) {
                        foreach ($harr["ports"] as $pid => $parr) {
                            $checkthissvc = false;
                            if (array_key_exists("selected", $parr)) {
                                $havesvc = true;
                                $checkthissvc = true;
                            }
                            // validate service name
                            if ($checkthishost == true && $checkthissvc == true) {
                                $servicename = $parr["servicename"];
                                if (is_valid_service_name($servicename) == false)
                                    $errmsg[$errors++] = _("Invalid service name") . " '<b>$servicename</b>' " . _("on host") . " <b>$hostname</b>";
                            }
                        }
                    }
                }
                if ($havehost == false)
                    $errmsg[$errors++] = _("No hosts were selected.");
                //$errmsg[$errors++]="Looks okay.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3OPTS:

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $job = grab_array_var($inargs, "job");
            $show = grab_array_var($inargs, "show", "");
            $addresstype = grab_array_var($inargs, "addresstype", "ip");
            $defaultservices = grab_array_var($inargs, "defaultservices", "common");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $services_u = json_decode(base64_decode($services_serial), true);
            $services_r = grab_array_var($inargs, "services", array());
            
            $services = array();
            if (is_array($services_u) && is_array($services_r))
                $services = array_merge($services_u, $services_r);


            $serviceargs = grab_array_var($inargs, "serviceargs");

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            echo '
                <input type="hidden" name="job" value="' . encode_form_val($job) . '">
                <input type="hidden" name="show" value="' . encode_form_val($show) . '">
                <input type="hidden" name="addresstype" value="' . encode_form_val($addresstype) . '">
                <input type="hidden" name="defaultservices" value="' . encode_form_val($defaultservices) . '">

                <input type="hidden" name="services_serial" value="' .
                        base64_encode(json_encode($services)) . '">
                <input type="hidden" name="serviceargs_serial" value="' .
                        base64_encode(json_encode($serviceargs)) . '">
            ';
            $output = ob_get_clean();
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETSTAGE4OPTS:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $serviceargs_serial = grab_array_var($inargs,
                "serviceargs_serial", "");

            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $services_u = json_decode(base64_decode($services_serial), true);
            $services_r = grab_array_var($inargs, "services", array());

            $services = array_merge($services_u, $services_r);

            $job = grab_array_var($inargs, "job");
            $show = grab_array_var($inargs, "show", "");
            $addresstype = grab_array_var($inargs, "addresstype", "ip");
            $defaultservices = grab_array_var($inargs, "defaultservices", "common");

            /*
            echo "SERVICES<BR>";
            print_r($services);
            echo "<BR>";
            echo "SERVICEARGS<BR>";
            print_r($serviceargs);
            echo "<BR>";
            */

            $objs = array();

            // process each host
            foreach ($services as $address => $arr) {
                // the host should be monitored...
                if (array_key_exists("selected", $arr)) {

                    $hostname = $arr["hostname"];

                    // add the host if necessary
                    if (!host_exists($hostname)) {

                        $hostaddress = $address;
                        if ($addresstype == "dns")
                            $hostaddress = @gethostbyaddr($address);

                        // add the host
                        $objs[] = array(
                            "type" => OBJECTTYPE_HOST,
                            "use" => "xiwizard_generic_host",
                            "host_name" => $hostname,
                            "address" => $hostaddress,
                            "_xiwizard" => $wizard_name,
                        );

                        // add a "Ping" service
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_genericnetdevice_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                    }

                    // process each port/service
                    foreach ($arr["ports"] as $pid => $parr) {

                        // skip this service if it wasn't selected
                        if (!array_key_exists("selected", $parr))
                            continue;

                        $servicename = $parr["servicename"];
                        $port = $parr["port"];
                        $protocol = strtolower($parr["protocol"]);

                        autodiscovery_configwizard_get_object_vars($port, $protocol, $use, $cmdline);

                        $newsvc = array(
                            'type' => OBJECTTYPE_SERVICE,
                            'host_name' => $hostname,
                            'service_description' => $servicename,
                            '_xiwizard' => $wizard_name,
                        );
                        if ($use != "")
                            $newsvc['use'] = $use;
                        if ($cmdline != "")
                            $newsvc['check_command'] = $cmdline;

                        $objs[] = $newsvc;
                    }
                }
            }

            //echo "OBJECTS:<BR>";

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;


        // THE FOLLOWING MODES ARE POST-CONFIGURATION CALLBACKS
        // THEY CAN BE USED TO DO CONFIGURATION TASKS, ETC AFTER A NEW
        //      CONFIGURATION HAS BEEN SUBMITTED

        case CONFIGWIZARD_MODE_COMMITERROR:
            break;

        case CONFIGWIZARD_MODE_COMMITCONFIGERROR:
            break;

        case CONFIGWIZARD_MODE_COMMITPERMSERROR:
            break;

        case CONFIGWIZARD_MODE_COMMITOK:

            break;

        default:
            break;
    }

    return $output;
}

/**
 * @param $port
 * @param $protocol
 * @param $use
 * @param $cmdline
 */
function autodiscovery_configwizard_get_object_vars($port, $protocol, &$use, &$cmdline)
{

    $use = "xiwizard_generic_service";
    $cmdline = "";

    $arr = array(
        "tcp" => array(
            21 => array("use" => "xiwizard_ftp_service"),
            22 => array("use" => "xiwizard_ssh_service"),
            25 => array("use" => "xiwizard_smtp_service"),
            80 => array("use" => "xiwizard_website_http_service"),
            110 => array("use" => "xiwizard_pop_service"),
            143 => array("use" => "xiwizard_imap_service"),
            443 => array("use" => "xiwizard_website_http_service", "cmdline" => "check_xi_service_http!-S"),
        ),
        "udp" => array(),
    );

    if (array_key_exists($port, $arr[$protocol])) {
        $match = $arr[$protocol][$port];
        $use = grab_array_var($match, "use");
        $cmdline = grab_array_var($match, "cmdline");
    } else {
        // use either xiwizard_tcp_service OR xiwizard_udp_service templates
        $use = "xiwizard_" . $protocol . "_service";
        // check_xi_service_tcp OR check_xi_service_udp
        $cmdline = "check_xi_service_" . $protocol . "!-p " . $port;
    }
}


/**
 * @param $service
 * @param $port
 * @param $protocol
 *
 * @return bool
 */
function autodiscovery_configwizard_is_common_service($service, $port, $protocol)
{

    $protoname = strtoupper($protocol);

    $name = $service;

    $common_services = array(
        "tcp" => array(
            21 => "FTP",
            22 => "SSH",
            23 => "Telnet",
            25 => "SMTP",
            80 => "HTTP",
            110 => "POP3",
            143 => "IMAP",
            389 => "LDAP",
            443 => "HTTPS",
            139 => "NetBIOS",
            631 => "IPP",
            993 => "IMAP SSL",
            3389 => "RDP",
            5666 => "NRPE",
            5667 => "NSCA",
        ),
        "udp" => array(),
    );

    if (array_key_exists($port, $common_services[$protocol]))
        return true;
    return false;
}


/**
 * @param $service
 * @param $port
 * @param $protocol
 *
 * @return string
 */
function autodiscovery_configwizard_get_friendly_service_name($service, $port, $protocol)
{

    $protoname = strtoupper($protocol);

    $name = $service;

    $friendly_names = array(
        "tcp" => array(
            21 => "FTP",
            22 => "SSH",
            23 => "Telnet",
            25 => "SMTP",
            80 => "HTTP",
            110 => "POP3",
            143 => "IMAP",
            389 => "LDAP",
            443 => "HTTPS",
            139 => "NetBIOS",
            631 => "IPP",
            993 => "IMAP SSL",
            3389 => "RDP",
            5666 => "NRPE",
            5667 => "NSCA",
        ),
        "udp" => array(),
    );

    if (array_key_exists($port, $friendly_names[$protocol]))
        $name = $friendly_names[$protocol][$port];
    else if ($service == "")
        $name = $protoname . " Port " . $port;
    else {
        // remome illegal chars in portnames -SW
        // `~!$%^&*|'"<>?,()=/\
        $badchars = explode(" ", "; ` ~ ! $ % ^ & * | ' \" < > ? , ( ) = / \\ { }");
        str_replace($badchars, " ", $service);
        $name = $protoname . " Port " . $port . " - " . $service . "";
    }
    return $name;
}