<?php
//
// DHCP Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

dhcp_configwizard_init();

function dhcp_configwizard_init()
{
    $name = "dhcp";

    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a DHCP server."),
        CONFIGWIZARD_DISPLAYTITLE => "DHCP",
        CONFIGWIZARD_FUNCTION => "dhcp_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "dhcp.png",
        CONFIGWIZARD_FILTER_GROUPS => array('network', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );

    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function dhcp_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "dhcp";

    // initialize return code and output
    $result = 0;
    $output = "";

    // initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address = nagiosccm_replace_user_macros($address);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");

            $ha = @gethostbyaddr($address);

            if (empty($ha)) {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);

            $server_ip = grab_array_var($inargs, "server_ip");
            $requested_address = grab_array_var($inargs, "requested_ip");
            $interface = grab_array_var($inargs, "interface");
            $mac_address = grab_array_var($inargs, "mac_address");
            $unicast = grab_array_var($inargs, "unicast");

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname_replaced = nagiosccm_replace_user_macros($hostname);
            $server_ip = grab_array_var($inargs, "server_ip");
            $unicast = grab_array_var($inargs, "unicast");

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname_replaced) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            if ($unicast != "" && $server_ip == "") {
                $errmsg[$errors++] = _("Unicast mode requires a server IP address to be specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");

            $server_ip = grab_array_var($inargs, "server_ip");
            $requested_address = grab_array_var($inargs, "requested_ip");
            $mac_address = grab_array_var($inargs, "mac_address");
            $interface = grab_array_var($inargs, "interface");
            $unicast = grab_array_var($inargs, "unicast");

            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="server_ip" value="' . encode_form_val($server_ip) . '">
        <input type="hidden" name="requested_ip" value="' . encode_form_val($requested_address) . '">
        <input type="hidden" name="mac_address" value="' . encode_form_val($mac_address) . '">
        <input type="hidden" name="interface" value="' . encode_form_val($interface) . '">
        <input type="hidden" name="unicast" value="' . encode_form_val($unicast) . '">
        <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
        <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">

            ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '<p>' . _('You will need to verify that iptables is allowing access to the dhcp ports (it doesn\'t by default) before this check will work.
            You can run the following command from the command prompt to enable the ports') . '</p>
            <p>iptables -I INPUT -i eth0 -p udp --dport 67:68 --sport 67:68 -j ACCEPT</p>';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $hostaddress = $address;

            $server_ip = grab_array_var($inargs, "server_ip");
            $requested_address = grab_array_var($inargs, "requested_ip");
            $mac_address = grab_array_var($inargs, "mac_address");
            $interface = grab_array_var($inargs, "interface");
            $unicast = grab_array_var($inargs, "unicast");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["server_ip"] = $server_ip;
            $meta_arr["requested_ip"] = $requested_address;
            $meta_arr["mac_address"] = $mac_address;
            $meta_arr["interface"] = $interface;
            $meta_arr["unicast"] = $unicast;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_genericnetdevice_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "network_node.png",
                    "statusmap_image" => "network_node.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // DHCP service
            $dhcpargs = "";
            if ($server_ip != "")
                $dhcpargs .= " -s " . $server_ip;
            if ($requested_address != "")
                $dhcpargs .= " -r " . $requested_address;
            if ($interface != "")
                $dhcpargs .= " -i " . $interface;
            if ($mac_address != "")
                $dhcpargs .= " -m " . $mac_address;
            if ($unicast != "")
                $dhcpargs .= " -u";

            $objs[] = array(
                "type" => OBJECTTYPE_SERVICE,
                "host_name" => $hostname,
                "service_description" => "DHCP",
                "use" => "xiwizard_generic_service",
                "check_command" => "check_dhcp!" . $dhcpargs,
                "_xiwizard" => $wizard_name,
            );

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


?>
