<?php
//
// Passive Check Config Wizard
// Copyright (c) 2011-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

passivecheck_configwizard_init();

function passivecheck_configwizard_init()
{
    $name = "passivecheck";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor passive service checks and events such as security alerts."),
        CONFIGWIZARD_DISPLAYTITLE => _("Passive Check"),
        CONFIGWIZARD_FUNCTION => 'passivecheck_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => 'passivecheck.png',
        CONFIGWIZARD_FILTER_GROUPS => array(''),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function passivecheck_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "passivecheck";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");
            $securitycheck = grab_array_var($inargs, "securitycheck", "no");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            $back = array_key_exists("backButton", $_POST);
            if ($back) break;

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address = nagiosccm_replace_user_macros($address);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");

            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);
            $securitycheck = grab_array_var($inargs, "securitycheck", "no");
            $isvolatile = grab_array_var($inargs, "isvolatile", "no");
            $statestalking = grab_array_var($inargs, "statestalking", "no");

            // Set defaults to yes if using securitycheck
            if ($securitycheck == "yes") {
                $isvolatile = "yes";
                $statestalking = "yes";
            }

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $services = json_decode(base64_decode($services_serial), true);

            if (!is_array($services)) {
                $services = grab_array_var($inargs, "services", array(1 => ($securitycheck == "yes") ? "Security Alert" : "Passive Service", 2 => "", 3 => "", 4 => "", 5 => ""));
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            $back = array_key_exists("backButton", $_POST);
            // if we hit back from the previous stage then we don't need to revalidate
            if ($back) break;

            // Get variables that were passed to us=
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $securitycheck = grab_array_var($inargs, "securitycheck", "no");
            $isvolatile = grab_array_var($inargs, "isvolatile");
            $statestalking = grab_array_var($inargs, "statestalking");

            $services = grab_array_var($inargs, "services");

            if (!is_array($services)) {
                $services_serial = grab_array_var($inargs, "services_serial", "");
                $services = json_decode(base64_decode($services_serial), true);
            }

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            if (!is_array($services)) {
                $errmsg[$errors++] = _("No service names specified.");
            } else if (count($services) == 0) {
                $errmsg[$errors++] = _("No service names specified.");
            } else {
                $havesvc = false;

                foreach ($services as $svc) {
                    if ($svc != "")
                        $havesvc = true;
                }

                if ($havesvc == false) {
                    $errmsg[$errors++] = _("No service names specified.");
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3OPTS:

            $output .= _("<div style='padding-bottom: 20px;'>There are no monitoring options to configure with passive checks.</div>");

            $result = CONFIGWIZARD_HIDE_OPTIONS;

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $securitycheck = grab_array_var($inargs, "securitycheck", "no");
            $isvolatile = grab_array_var($inargs, "isvolatile");
            $statestalking = grab_array_var($inargs, "statestalking");

            $services = grab_array_var($inargs, "services");
            if (!is_array($services)) {
                $services_serial = grab_array_var($inargs, "services_serial", "");
                $services = json_decode(base64_decode($services_serial), true);
            }

            $serviceargs = grab_array_var($inargs, "serviceargs");

            $output = '

        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="securitycheck" value="' . encode_form_val($securitycheck) . '">
        <input type="hidden" name="isvolatile" value="' . encode_form_val($isvolatile) . '">
        <input type="hidden" name="statestalking" value="' . encode_form_val($statestalking) . '">
        <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
        <input type="hidden" name="serviceargs_serial" value="' .base64_encode(json_encode($serviceargs)) . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETSTAGE4OPTS:

            // hide some notification options
            $output = '';
            $result = CONFIGWIZARD_HIDE_OPTIONS;
            $outargs[CONFIGWIZARD_HIDDEN_OPTIONS] = array(
                CONFIGWIZARD_HIDE_NOTIFICATION_DELAY,
                CONFIGWIZARD_HIDE_NOTIFICATION_INTERVAL,
            );

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $securitycheck = grab_array_var($inargs, "securitycheck", "no");
            $isvolatile = grab_array_var($inargs, "isvolatile");
            $statestalking = grab_array_var($inargs, "statestalking");

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_passive_host",
                    "host_name" => $hostname,
                    "address" => $address,
                    'stalking_options' => ($statestalking == "yes") ? "o,u,d" : "n",
                    "_xiwizard" => $wizard_name,
                );
            }

            foreach ($services as $svcname) {

                if (trim($svcname) == "")
                    continue;

                //echo "PROCESSING: $hostname -> $hoststate<BR>\n";

                $objs[] = array(
                    'type' => OBJECTTYPE_SERVICE,
                    'host_name' => $hostname,
                    'service_description' => $svcname,
                    'use' => 'xiwizard_passive_service',
                    'check_interval' => 1,
                    'retry_interval' => 1,
                    'max_check_attempts' => 1,
                    'is_volatile' => ($isvolatile == "yes") ? 1 : 0,
                    'stalking_options' => ($statestalking == "yes") ? "o,w,u,c" : "n",
                    //'icon_image' => 'passivecheck.png',
                    '_xiwizard' => $wizard_name,
                );
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;


        // THE FOLLOWING MODES ARE POST-CONFIGURATION CALLBACKS
        // THEY CAN BE USED TO DO CONFIGURATION TASKS, ETC AFTER A NEW
        //      CONFIGURATION HAS BEEN SUBMITTED

        case CONFIGWIZARD_MODE_COMMITERROR:
            echo "COMMITERROR!\n";
            break;

        case CONFIGWIZARD_MODE_COMMITCONFIGERROR:
            echo "COMMITCONFIGERROR!\n";
            break;

        case CONFIGWIZARD_MODE_COMMITPERMSERROR:
            echo "COMMITPERMSERROR!\n";
            break;

        case CONFIGWIZARD_MODE_COMMITOK:

            $services_serial = grab_array_var($inargs, "services_serial");
            $services = json_decode(base64_decode($services_serial), true);

            break;

        default:
            break;
    }

    return $output;
}
