<?php
//
// Website Defacement Config Wizard
// Copyright (c) 2008-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

website_defacementwizard_init();

function website_defacementwizard_init()
{
    $name = "website_defacement";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a website for defacement."),
        CONFIGWIZARD_DISPLAYTITLE => _("Website Defacement"),
        CONFIGWIZARD_FUNCTION => "website_defacementwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "website_defacement.png",
        CONFIGWIZARD_FILTER_GROUPS => array('website'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param        $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function website_defacementwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    // Grab core wizard info
    $wizard_name = grab_array_var($inargs, "wizard");
    $wiz_args = get_configwizard_by_name($wizard_name);
    $wizard_title = isset($wiz_args[CONFIGWIZARD_DISPLAYTITLE]) ? $wiz_args[CONFIGWIZARD_DISPLAYTITLE] : '';

    // Gotta have a sesh
    if (!isset($_POST['wizard'])) {
        $_SESSION[$wizard_name] = [];
    }

    // Give session a simple name for convenience
    $wiz_sesn =& $_SESSION[$wizard_name];

    // $inargs keys that needn't/can't be encoded with htmlentities()
    $noEncodeList = [];

    // Array elements that may have user macros
    $userMacroKeys = [];

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $url = grab_in_var($inargs, "url", "http://", $wiz_sesn);

            if (isset($_POST['backButton'])) {
                $hostname = grab_in_var($inargs, "hostname", "", $wiz_sesn);
                $servicename = grab_in_var($inargs, "servicename", "", $wiz_sesn);
                $port = grab_in_var($inargs, "port", "", $wiz_sesn);
                $ssl = grab_in_var($inargs, "ssl", "", $wiz_sesn);
                $basicauth = grab_in_var($inargs, "basicauth", "", $wiz_sesn);
                $username = grab_in_var($inargs, "username", "", $wiz_sesn);
                $password = grab_in_var($inargs, "password", "", $wiz_sesn);
                $onredirect = grab_in_var($inargs, "onredirect", "=", $wiz_sesn);
                $services = grab_in_var($inargs, "services", "", $wiz_sesn);
                $serviceargs = grab_in_var($inargs, "serviceargs", "", $wiz_sesn);
            }

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $url = grab_in_var($inargs, "url", "", $wiz_sesn);
            $url = nagiosccm_replace_user_macros($url);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($url) == false) {
                $errmsg[$errors++] = _("No URL specified.");
            } else if (!valid_url($url)) {
                $errmsg[$errors++] = _("Invalid URL.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $url = grab_in_var($inargs, "url", "", $wiz_sesn);
            $urlparts = parse_url($url);
            $hostname = grab_array_var($urlparts, "host");
            $urlscheme = grab_array_var($urlparts, "scheme");
            $port = grab_array_var($urlparts, "port");
            $username = grab_array_var($urlparts, "user");
            $password = grab_array_var($urlparts, "pass");

            if ($urlscheme == "https") {
                $ssl = "on";
            } else {
                $ssl = "off";
            }

            if ($port == "") {
                if ($ssl == "on") {
                    $port = 443;
                } else {
                    $port = 80;
                }
            }

            $basicauth = "";

            if ($username != "") {
                $basicauth = "on";
            }

            $ip = gethostbyname($hostname);

            $hostname = grab_in_var($inargs, "hostname", $hostname, $wiz_sesn);
            $servicename = grab_in_var($inargs, "servicename", websitedeface_configwizard_url_to_name($url), $wiz_sesn);
            $port = grab_in_var($inargs, "port", $port, $wiz_sesn);
            $ssl = grab_in_var($inargs, "ssl", $ssl, $wiz_sesn);
            $basicauth = grab_in_var($inargs, "basicauth", $basicauth, $wiz_sesn);
            $username = grab_in_var($inargs, "username", $username, $wiz_sesn);
            $password = grab_in_var($inargs, "password", $password, $wiz_sesn);
            $onredirect = grab_in_var($inargs, "onredirect", "ok", $wiz_sesn);

            $service_defaults = [
                "httpcontent" => "on",
                "httpregex" => "off",
                "regexinvert" => "off",
                "cinput" => "off"
            ];

            $serviceargs_defaults = [
                "httpcontentstr" => "",
                "httpregexstr" => ""
            ];

            $services = grab_in_var($inargs, "services", $service_defaults, $wiz_sesn);
            $serviceargs = grab_in_var($inargs, "serviceargs", $serviceargs_defaults, $wiz_sesn);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $url = grab_in_var($inargs, "url", "", $wiz_sesn);
            $hostname = grab_in_var($inargs, "hostname", "", $wiz_sesn);
            $servicename = grab_in_var($inargs, "servicename", "", $wiz_sesn);
            $port = grab_in_var($inargs, "port", "", $wiz_sesn);
            $username = grab_in_var($inargs, "username", "", $wiz_sesn);
            $password = grab_in_var($inargs, "password", "", $wiz_sesn);
            $services = grab_in_var($inargs, "services", [], $wiz_sesn);
            $serviceargs = grab_in_var($inargs, "serviceargs", [], $wiz_sesn);


            $hostname = nagiosccm_replace_user_macros($hostname);
            $port = nagiosccm_replace_user_macros($port);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            if (is_valid_service_name($servicename) == false) {
                $errmsg[$errors++] = _("Invalid service prefix.  Can only contain alphanumeric characters, spaces, and the following: <b>.\:_-</b>");
			}

            if (have_value($url) == false) {
                $errmsg[$errors++] = _("No URL specified.");
            } else if (!valid_url($url)) {
                $errmsg[$errors++] = _("Invalid URL.");
            }

            if (array_key_exists("httpcontent", $services)) {
                if (array_key_exists("httpcontentstr", $serviceargs)) {
                    if (have_value($serviceargs["httpcontentstr"]) == false) {
                        $errmsg[$errors++] = _("You must specify a string to expect in the web page content.");
                    }
                }
            }

            if (array_key_exists("httpregex", $services)) {
                if (array_key_exists("httpregexstr", $serviceargs)) {
                    if (have_value($serviceargs["httpregexstr"]) == false)
                        $errmsg[$errors++] = _("You must specify a regular expression to expect in the web page content.");
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $url = grab_in_var($inargs, "url", "", $wiz_sesn);
            $servicename = grab_in_var($inargs, "servicename", "", $wiz_sesn);
            $hostname = grab_in_var($inargs, "hostname", "", $wiz_sesn);
            $ip = grab_in_var($inargs, "ip", "", $wiz_sesn);
            $ssl = grab_in_var($inargs, "ssl", "", $wiz_sesn);
            $port = grab_in_var($inargs, "port", "", $wiz_sesn);
            $username = grab_in_var($inargs, "username", "", $wiz_sesn);
            $password = grab_in_var($inargs, "password", "", $wiz_sesn);
            $onredirect = grab_in_var($inargs, "onredirect", "", $wiz_sesn);

            $services = grab_in_var($inargs, "services", "", $wiz_sesn);
            $serviceargs = grab_in_var($inargs, "serviceargs", "", $wiz_sesn);
            $httpcontentstr = grab_in_var($inargs, "httpcontentstr", "", $wiz_sesn);
            $httpregexstr = grab_in_var($inargs, "httpregexstr", "", $wiz_sesn);

            $output = '

            ';

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_in_var($inargs, "hostname", "", $wiz_sesn);
            $servicename = grab_in_var($inargs, "servicename", "", $wiz_sesn);
            $ip = grab_in_var($inargs, "ip", "", $wiz_sesn);
            $url = grab_in_var($inargs, "url", "", $wiz_sesn);
            $ssl = grab_in_var($inargs, "ssl", "", $wiz_sesn);
            $port = grab_in_var($inargs, "port", "", $wiz_sesn);
            $username = grab_in_var($inargs, "username", "", $wiz_sesn);
            $password = grab_in_var($inargs, "password", "", $wiz_sesn);
            $onredirect = grab_in_var($inargs, "onredirect", "", $wiz_sesn);

            $services = grab_in_var($inargs, "services", "", $wiz_sesn);
            $serviceargs = grab_in_var($inargs, "serviceargs", "", $wiz_sesn);

            $urlparts = parse_url($url);
            $hostaddress = $urlparts["host"];

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["servicename"] = $servicename;
            $meta_arr["ip"] = $ip;
            $meta_arr["url"] = $url;
            $meta_arr["ssl"] = $ssl;
            $meta_arr["port"] = $port;
            $meta_arr["username"] = $username;
            $meta_arr["password"] = $password;
            $meta_arr["onredirect"] = $onredirect;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $regexinvert = "";
            if (array_key_exists("regexinvert", $services) && $services["regexinvert"] == 'on') {
                $regexinvert = "!--invert-regex";
            }

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_website_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "website_defacement.png",
                    "statusmap_image" => "website_defacement.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            $pluginopts = "";

            $vhost = $urlparts["host"];

            $pluginopts .= " -f " . $onredirect; // on redirect, follow (OK status)
            if (!empty($ip)) {
                $pluginopts .= " -I " . $ip; // ip address
            }

            $urlpath = grab_array_var($urlparts, "path", "/");

            if ($ssl == "on") {
                $pluginopts .= " -S --sni";
            }
            if ($port != "") {
                $pluginopts .= " -p " . $port;
            }
            if ($username != "") {
                $pluginopts .= " -a \"" . $username . ":" . $password . "\"";
            }

            // Need to add query (after ?) and fragment (after #) back on -JO

            if (!empty($urlparts["query"])) {
                $urlpath .= "?" . $urlparts["query"];
            }

            if (!empty($urlparts["fragment"])) {
                $urlpath .= "#" . $urlparts["fragment"];
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                // echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($svc) {

                    case "http":
                        $extra = " -u " . escapeshellarg($urlpath);
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "HTTP",
                            "use" => "xiwizard_check_deface_host",
                            "check_command" => "check-host-alive-http!-H " . escapeshellarg($url) . " " . $pluginopts . $extra,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "httpcontent":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $servicename . " Website Defacement",
                            "use" => "xiwizard_check_deface_service",
                            "check_command" => "check_xi_deface!" . $serviceargs["httpcontentstr"] . "!" . $urlpath . "! --invert-regex" . $pluginopts,
                            "check_interval" => 60,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "httpregex":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $servicename . " Web Page Regex Match",
                            "use" => "xiwizard_check_deface_service",
                            "check_command" => "check_xi_deface!" . $serviceargs["httpregexstr"] . "!" . $urlpath . "!" . $regexinvert . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    default:
                        break;
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}

/**
 * @param $url
 *
 * @return mixed|string
 */
function websitedeface_configwizard_url_to_name($url)
{

    $urlparts = parse_url($url);
    $path = grab_array_var($urlparts, "path", "");

    $path = str_replace("/", "_", $path);
    $path = str_replace("\\", "_", $path);
    $path = str_replace("?", "_", $path);
    $path = str_replace(";", "_", $path);
    $path = str_replace("&", "_", $path);
    $path = str_replace(":", "_", $path);

    if ($path == "")
        $path = "_";

    return $path;
}