<?php
//
// Slack Notifications - Event Handler Config Wizard
// Copyright (c) 2023 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

slacknotifications_configwizard_init();

function slacknotifications_configwizard_init()
{
    $name = "eventhandler_notify_slack";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "1.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Send Host/Service updates to Slack."),
        CONFIGWIZARD_DISPLAYTITLE => _("Slack Notifications"),
        CONFIGWIZARD_FUNCTION => "slacknotifications_configwizard_func",
        CONFIGWIZARD_REQUIRES_VERSION => 60030,
        CONFIGWIZARD_FILTER_GROUPS => array('notifications'),
        CONFIGWIZARD_PREVIEWIMAGE => "../../../../../configwizards/eventhandler_notify_slack/logos/Slack-mark.png"
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param        $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string $output
 */
function slacknotifications_configwizard_func($mode, $inargs, &$outargs, &$result)
{
    global $debug;
    $debug = true;

    $wizard_name = "eventhandler_notify_slack";
    if(empty($mode)) {
        $mode = "";
    }

    // Initialize return code and output
    $result = 0;

    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

// Select Hosts/Services
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            $nextstep = encode_form_val(grab_array_var($_POST,'nextstep',false),ENT_QUOTES);
            if ($nextstep == '') {
                unset($_SESSION[$wizard_name]); 
            }

            // Stage 1 HTML
            $output = '
            <link rel=\'stylesheet\' type=\'text/css\' href="../includes/components/ccm/css/style.css" />
            <style>
                body table {
                    font-size: 1.1rem;
                }
            </style>
            <h5 class="ul">' . _('Setup Instructions') . '</h5>
                <ul style="padding: 0 0 0 30px;">
                    <li><a href="https://assets.nagios.com/downloads/nagiosxi/docs/Slack-Notification-Setup.pdf" target="_blank">' . _('View Setup Instructions PDF') . '</a> in a new tab.</li>
                </ul>
            <h5 class="ul">' . _('Configuration Information') . '</h5>
            <table class="table table-condensed table-no-border table-auto-width">
                <tr>
                    <td class="vt">
                        <label>' . _('Webhook URL') . ':</label>
                    </td>
                    <td>
                        <input type="text" size="60" name="url" id="url" class="form-control">
                        <div class="subtext">' . _('Paste the Slack Webhook URL here.') . '</div>
                    </td>
                </tr>
                <tr>
                    <td class="vt">
                        <label>' . _('Choose Hosts/Services') . ':</label>
                    </td>
                    <td>
                        <button type="button" class="btn btn-sm btn-info btn-hostBox" onclick="show_overlay(\'host\')">' . _('Select Hosts') . '<span class="badge">0</span></button>
                        <button type="button" class="btn btn-sm btn-info btn-serviceBox" onclick="show_overlay(\'service\')">' . _('Select Services') . '<span class="badge">0</span></button>
                        <div class="subtext">' . _('Add the Slack notification event handler to the selected Hosts/Services:') . '</div>
                    </td>
            </table>';


            // Stage 1 Host Overlay
            $hostxml = get_xml_host_objects();


            $display_host_display_name = grab_request_var("display_host_display_name", get_user_meta(0, "display_host_display_name"));
            $display_service_display_name = grab_request_var("display_service_display_name", get_user_meta(0, "display_service_display_name"));

            // Stage 1 Service Overlay
            $servicexml = get_xml_service_objects();
            $servicexml_min = array();
            foreach($servicexml as $key => $service) {
                $servicexml_min_out = array();
                foreach($service as $key => $value) {
                    $value = strval($value);

                    if($key == 'host_name') {
                        $servicexml_min_out['host_name'] = $value;
                    } else if($key == 'service_description' && !$display_service_display_name) {
                        $servicexml_min_out['name'] = $value;
                    } else if($key == 'display_name' && $display_service_display_name) {
                        $servicexml_min_out['name'] = $value;
                    }
                }
                $servicexml_min[] = $servicexml_min_out;
            }

            // hidden overlay divs
            $output .= construct_overlay("host", $hostxml);
            $output .= construct_overlay("service", $servicexml);

            // var hostxml = '" . json_encode($hostxml) . "';
            // var servicexml = '" . json_encode($servicexml_min) . "';

            // overlay js
            $output .= "<script>

            function show_overlay(div)
            {
                var ID = '#'+div+'Box';

                // Check if overlay actually exists 
                if ($(ID).html() == null) {
                    alert('Undefined overlay '+div+'!');
                    return;
                }

                // Cover the page with a opaque hidden div...
                var height = $(window).height() + 20;
                var width = $(window).width() + 20;

                // whiteout();

                // Make the overlay width much bigger if it's a small screen
                if ($(window).width() < 1024) {
                    var overlay_width = width-(width*0.10);
                } else {
                    var overlay_width = width-(width*0.16);
                }
                var overlay_height = height-(height*0.22);

                // Set defaults (min and max)
                if (overlay_width < 400) { overlay_width = 400; }

                // Display the selection overlay
                $(ID).css('width', overlay_width)
                    .css('position', 'absolute')
                    .css('height', overlay_height)
                    .center().show();

                $(ID).find('select').val([]);

                var filter_height = $(ID).find('.listDiv .filter').outerHeight();
                var title_height = $(ID).find('.overlay-title').outerHeight();
                var padding_height = 45;
                var bottom_height = $(ID).find('.overlay-left-bottom').outerHeight();
                var close_height = $(ID).find('.closeOverlay').outerHeight();

                // Resize selection box to be most of the height (use as much real estate as possible)
                var select_height = overlay_height - filter_height - title_height - padding_height - bottom_height - close_height;
                $(ID).find('.lists').css('height', Math.round(select_height));

                var assigned_height = overlay_height - title_height - padding_height - 28;

                $(ID).find('.assigned-container').css('max-height', Math.round(assigned_height));

                $(document).keyup(function(e) {
                    if (e.keyCode == 27) {
                        killOverlay(div);
                    }
                });
            }
            </script>";
            break;

// Validate - Select Hosts/Services
        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            $url = grab_array_var($inargs, "url");
            $url = nagiosccm_replace_user_macros($url);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($url) == false)
                $errmsg[$errors++] = _("No Webhook URL specified.");
            else if (!valid_url($url))
                $errmsg[$errors++] = _("Invalid URL.");

            // if (have_value($host) == false) {
            //     $errmsg[$errors++] = _("No template host specified.");
            // } else if (host_exists($host) == false) {
            //     $errmsg[$errors++] = _("Template host could not be found.");
            // }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;

// Verify Selection from Stage 1
        case CONFIGWIZARD_MODE_GETSTAGE2HTML:
            $url = grab_array_var($inargs, "url");
            $hosts = grab_array_var($inargs, "hosts");
            $services = grab_array_var($inargs, "services");

            $output = '';

            $output .= '
                <style>
                    thead, tbody {
                        display: block;
                    }
                    
                    tbody {
                        max-height: 50rem; /* Set to desired height */
                        width: 100%;
                        overflow-y: auto;
                        overflow-x: hidden;
                    }
                </style>
                <h5 class="ul">' . _('Verify Selection') . ':</h5> 
                <table class="table table-condensed table-no-border table-auto-width" disabled>
                    <thead>
                    </thead>
                    <tbody>
                        <tr>
                            <td class="vt">
                                <label>' . _('Webhook URL') . ':</label>
                            </td>
                            <td>
                                <input type="text" size="60" name="url" id="url" class="form-control" value="' . $url . '" disabled>
                            </td>
                        </tr>';
                if (!empty($hosts)){
                    $output .= '
                        <tr>
                            <td class="vt">
                                <label>' . _('Hosts') . ':</label>
                            </td>
                            <td>';
                            foreach($hosts as $host) {
                                $host = json_decode($host, true);
                                $output .= '<input style="display:block" class="form-control" name="hosts[]" value="'. $host['host_name'] . '" disabled>';
                            }
                    $output .= '</td>
                        </tr>';
                }
                if (!empty($services)){
                    $output .= '
                        <tr>
                            <td class="vt">
                                <label>' . _('Services') . ':</label>
                            </td>
                            <td>';
                            foreach($services as $service) {
                                $service = json_decode($service, true);
                                $output .= '<div>
                                                <input style="display:inline-block" class="form-control" value="' . $service['host_name'] . '" disabled>';
                                $output .= '    <input style="display:inline-block" class="form-control" value="' . $service['display_name'] . '" disabled>
                                            </div>';
                            }
                    $output .= '</td>
                        </tr>';
                }
                    
                $output .= '
                    </tbody>
                </table>';

                $output .= '<input type="hidden" id="url" name="url" value="' . $url . '">';
                $output .= "<input type='hidden' id='hosts' name='hosts' value='" . json_encode($hosts) . "'>";
                $output .= "<input type='hidden' id='services' name='services' value='" . json_encode($services) . "'>";
            break;

// Validate stage 2 - not really needed
        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            $url = grab_array_var($inargs, "url");
            $hosts = json_decode(grab_array_var($inargs, "hosts"));
            $services = json_decode(grab_array_var($inargs, "services"));

            $_SESSION[$wizard_name]['url'] = $url;
            $_SESSION[$wizard_name]['hosts'] = $hosts;
            $_SESSION[$wizard_name]['services'] = $services;

            break;

// Message Customization?
        // case CONFIGWIZARD_MODE_GETSTAGE3HTML:
        //     break;

        // case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:
        //     break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:
            $output = '';
            break;

// Define Host/Service Objects
        case CONFIGWIZARD_MODE_GETOBJECTS:
            $url = encode_form_val($_SESSION[$wizard_name]['url']);
            $hosts = $_SESSION[$wizard_name]['hosts'];
            $services = $_SESSION[$wizard_name]['services'];

            foreach(array('hosts', 'services') as $type) {
                if(empty($$type)) {
                    $$type = array();
                }
            }

            $hosteventhandlerargument = '"$HOSTNAME$" "$HOSTADDRESS$" "$HOSTSTATE$" "$HOSTOUTPUT$" "$LONGDATETIME$"';
            $serviceeventhandlerargument = '"$HOSTNAME$" "$HOSTADDRESS$" "$SERVICESTATE$" "$SERVICEOUTPUT$" "$LONGDATETIME$" "$SERVICEDESC$"';

            $objs = array();
            $host_command_name = "host_slack_curl_".uniqid();
            $objs[] = array(
                "type" => OBJECTTYPE_COMMAND,
                "command_name" => "$host_command_name",
                "command_line" => '/usr/local/nagiosxi/scripts/slack_curl.sh \'' . $url . "' host $hosteventhandlerargument",
            );

            $service_command_name = "service_slack_curl_".uniqid();
            $objs[] = array(
                "type" => OBJECTTYPE_COMMAND,
                "command_name" => "$service_command_name",
                "command_line" => '/usr/local/nagiosxi/scripts/slack_curl.sh \'' . $url . "' service $serviceeventhandlerargument",
            );

            foreach($hosts as $host) {
                $host = json_decode($host, true);
                $host_name = $host['host_name'];
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "host_name" => encode_form_val($host_name),
                    "event_handler" => "$host_command_name",
                    "event_handler_enabled" => 1,
                );
            }

            foreach($services as $service) {
                $service = json_decode($service, true);
                $host_name = $service['host_name'];
                $service_description = $service['service_description'];
                $objs[] = array(
                    "type" => OBJECTTYPE_SERVICE,
                    "host_name" => encode_form_val($host_name),
                    "service_description" => encode_form_val($service_description),
                    "event_handler" => "$service_command_name",
                    "event_handler_enabled" => 1,
                );
            }

            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}