<?php
//
// Mountpoint Config Wizard
// Copyright (c) 2016-2021 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__).'/../configwizardhelper.inc.php');

// run the initialization function
mountpoint_configwizard_init();

function mountpoint_configwizard_init()
{
    $name = "mountpoint";
    
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.0",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a NFS, CIFS or DAVFS mountpoint."),
        CONFIGWIZARD_DISPLAYTITLE => _("Mountpoint"),
        CONFIGWIZARD_FUNCTION => "mountpoint_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "mountpoint.png",
        CONFIGWIZARD_FILTER_GROUPS => array('linux','otheros'),
        CONFIGWIZARD_REQUIRES_VERSION => 500
        );

    register_configwizard($name,$args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function mountpoint_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "mountpoint";
    
    // initialize return code and output
    $result = 0;
    $output = "";
    
    // initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $service_description = grab_array_var($inargs, "service_description", "_");
            $mtab = grab_array_var($inargs, 'mtab', "/proc/mounts");
            $fstab = grab_array_var($inargs, 'fstab', "/etc/fstab");
            $FSF = grab_array_var($inargs, 'FSF', "3");
            $MF = grab_array_var($inargs, 'MF', "2");
            $OF = grab_array_var($inargs, 'OF', "4");
            $rtime = grab_array_var($inargs, 'rtime', "3");
            $softlink = grab_array_var($inargs, 'softlink');
            $ignorefs = grab_array_var($inargs, 'ignorefs');
            $auto = grab_array_var($inargs, 'auto');
            $write = grab_array_var($inargs, 'write');

            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:
            $errors = 0;
            $errmsg = array();

            if (!file_exists("/usr/local/nagios/libexec/check_mountpoints.sh")) {
                $errmsg[$errors++] = _('It looks like you are missing check_mountpoints.sh on your Nagios XI server. To use this wizard you must install the check_mountpoints.sh plugin on your server located in the this wizards plugin directory here: /usr/local/nagios/libexec/');
            }
            if ($errors>0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES]=$errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // get variables that were passed to us
            $service_description = grab_array_var($inargs, "service_description");
            $mtab = grab_array_var($inargs, 'mtab');
            $fstab = grab_array_var($inargs, 'fstab');
            $FSF = grab_array_var($inargs, 'FSF');
            $MF = grab_array_var($inargs, 'MF');
            $OF = grab_array_var($inargs, 'OF');
            $rtime = grab_array_var($inargs, 'rtime');
            $softlink = grab_array_var($inargs, 'softlink');
            $ignorefs = grab_array_var($inargs, 'ignorefs');
            $auto = grab_array_var($inargs, 'auto');
            $write = grab_array_var($inargs, 'write');

            $scansuccess = 1;
            $mounts = array();

            $mountpoint_cmd = "/usr/local/nagios/libexec/check_mountpoints.sh -a";
            exec($mountpoint_cmd, $mount_output, $mount_return);

            if(!empty($mount_output)) {
                $mount_data = $mount_output[0];

                // parse response
                $response = preg_split("/:/", $mount_data);

                // if plugin returns OK
                if ($response[0] == "OK") {
                    preg_match("/\((.*?)\)$/", $mount_data, $mount_container);
                    $mount_data = explode(" ", $mount_container[1]);
                    $mount_data = array_filter($mount_data, 'strlen');

                    foreach ($mount_data as $key => $val) {
                        array_push($mounts, '"' . $val . '"');
                    }
                } else if ($response[0] == "CRITICAL") { // if plugin returns critical
                    preg_match_all("/:\s(.*?)\s;$/", $mount_data, $mount_container);
                    $mount_data = explode(" ; ", $mount_container[1][0]);

                    foreach ($mount_data as $key => $val) {
                        array_push($mounts, '"' . $val . '"');
                    }
                }

                // prepare for Javascript array
                $mounts = implode(",", $mounts);
            }

            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            // get variables that were passed to us
            $service_description = grab_array_var($inargs, "service_description");
            $mtab = grab_array_var($inargs, 'mtab');
            $fstab = grab_array_var($inargs, 'fstab');
            $FSF = grab_array_var($inargs, 'FSF');
            $MF = grab_array_var($inargs, 'MF');
            $OF = grab_array_var($inargs, 'OF');
            $rtime = grab_array_var($inargs, 'rtime');
            $auto = grab_array_var($inargs, 'auto');
            $mount = grab_array_var($inargs, "mount");

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_service_name($service_description) == false) {
                $errmsg[$errors++] = _("Invalid service prefix.  Can only contain alphanumeric characters, spaces, and the following:") . "<b>.\:_-</b>";
            }
            if (!is_numeric($FSF)) {
                $errmsg[$errors++] = _("FS Field number is not numeric.");
            } 
            if (!is_numeric($MF)) {
                $errmsg[$errors++] = _("Mount Field number is not numeric.");
            }
            if (!is_numeric($OF)) {
                $errmsg[$errors++] = _("Option Field number is not numeric.");
            }
            if (!is_numeric($rtime)) {
                $errmsg[$errors++] = _("Response time number is not numeric.");
            }
            if (count($mount) == 1 && $mount[0] == "" && $auto == "") {
                $errmsg[$errors++] = _("No mounts selected and auto select isn't enabled. Either add mounts or enable auto select.");
            }

            if($errors>0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:
        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            // get variables that were passed to us
            $service_description = grab_array_var($inargs, "service_description");
            $mtab = grab_array_var($inargs, 'mtab');
            $fstab = grab_array_var($inargs, 'fstab');
            $FSF = grab_array_var($inargs, 'FSF');
            $MF = grab_array_var($inargs, 'MF');
            $OF = grab_array_var($inargs, 'OF');
            $rtime = grab_array_var($inargs, 'rtime');
            $softlink = grab_array_var($inargs, 'softlink');
            $ignorefs = grab_array_var($inargs, 'ignorefs');
            $auto = grab_array_var($inargs, 'auto');
            $write = grab_array_var($inargs, 'write');
            $mount = grab_array_var($inargs, 'mount');

            $mount_serial = base64_encode(json_encode($mount));

            $output = '
                <input type="hidden" name="service_description" value="' .  encode_form_val($service_description) . '" />
                <input type="hidden" name="mtab" value="' .  encode_form_val($mtab) . '" />
                <input type="hidden" name="fstab" value="' .  encode_form_val($fstab) . '" />
                <input type="hidden" name="FSF" value="' .  encode_form_val($FSF) . '" />
                <input type="hidden" name="MF" value="' .  encode_form_val($MF) . '" />
                <input type="hidden" name="OF" value="' .  encode_form_val($OF) . '" />
                <input type="hidden" name="rtime" value="' .  encode_form_val($rtime) . '" />
                <input type="hidden" name="softlink" value="' .  encode_form_val($softlink) . '" />
                <input type="hidden" name="ignorefs" value="' .  encode_form_val($ignorefs) . '" />
                <input type="hidden" name="auto" value="' .  encode_form_val($auto) . '" />
                <input type="hidden" name="write" value="' .  encode_form_val($write) . '" />
                <input type="hidden" name="mount" value="' . encode_form_val($mount_serial) . '" />
            ';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $service_description = grab_array_var($inargs, "service_description");
            $mtab = grab_array_var($inargs, 'mtab');
            $fstab = grab_array_var($inargs, 'fstab');
            $FSF = grab_array_var($inargs, 'FSF');
            $MF = grab_array_var($inargs, 'MF');
            $OF = grab_array_var($inargs, 'OF');
            $rtime = grab_array_var($inargs, 'rtime');
            $softlink = grab_array_var($inargs, 'softlink');
            $ignorefs = grab_array_var($inargs, 'ignorefs');
            $auto = grab_array_var($inargs, 'auto');
            $write = grab_array_var($inargs, 'write');
            $mount = grab_array_var($inargs, 'mount');

            $mount = json_decode(base64_decode($mount), true);
            $hostname = "localhost";

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["service_description"] = $service_description;
            $meta_arr["mtab"] = $mtab;
            $meta_arr["fstab"] = $fstab;
            $meta_arr["FSF"] = $FSF;
            $meta_arr["MF"] = $MF;
            $meta_arr["OF"] = $OF;
            $meta_arr["rtime"] = $rtime;
            $meta_arr["softlink"] = $softlink;
            $meta_arr["ignorefs"] = $ignorefs;
            $meta_arr["auto"] = $auto;
            $meta_arr["write"] = $write;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if(!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_generic_host",
                    "host_name" => $hostname,
                    "icon_image" => "mountpoint.png",
                    "statusmap_image" => "mountpoint.png",
                    "_xiwizard" => $wizard_name,
                );
            }
            $extramountargs = "";

            if ($softlink == "on") {
                $extramountargs .= "-L ";
            }
            if ($ignorefs == "on") {
                $extramountargs .= "-i ";
            }
            if ($auto == "on") {
                $extramountargs .= "-a ";
            }
            if ($write == "on") {
                $extramountargs .= "-w ";
            }

            foreach($mount as $mnt) {
                $extramountargs .= "$mnt ";
            }

            $objs[] = array(
                "type" => OBJECTTYPE_SERVICE,
                "host_name" => $hostname,
                "service_description" => $service_description,
                "use" => "xiwizard_mountpoint_check",
                "check_command" => "check_mountpoint!$mtab!$fstab!$FSF!$MF!$OF!$rtime!$extramountargs",
                "icon_image" => "mountpoint.png",
                "_xiwizard" => $wizard_name,
            );

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS]=$objs;

            break;

        default:
            break;          
        }

    return $output;
}
?>