<?php
//
// MSSQL Database Config Wizard
// Copyright (c) 2010-2025 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__).'/../configwizardhelper.inc.php');

mssqldatabase_configwizard_init();

// NOTE: This wizard now uses the check_mssql_server.php plugin (under the covers).
function mssqldatabase_configwizard_init()
{
    $name = "mssql_database";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "3.2.0",
        CONFIGWIZARD_IS_PREMIUM => true,
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a MSSQL Database"),
        CONFIGWIZARD_DISPLAYTITLE => _("MSSQL Database"),
        CONFIGWIZARD_FUNCTION => "mssqldatabase_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "mssqldatabase.png",
        CONFIGWIZARD_FILTER_GROUPS => array('windows','database'),
        CONFIGWIZARD_REQUIRES_VERSION => 70000
    );
    register_configwizard($name, $args);
}


/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
# TODO: required params are now required to be listed before optional params.  Setting defaults works for now. - FIX!!
function mssqldatabase_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    global $configwizards;
    $wizard_name = "mssql_database";
    $wizard_title = $configwizards[$wizard_name]['display_title'];

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "1433");
            $instance = grab_array_var($inargs, "instance", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $instancename = grab_array_var($inargs, "instancename", "master");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            # These will only be set by going back from step 2.
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");
            $config_serial = grab_array_var($inargs, "config_serial");

            $config = (!empty($config_serial)) ? json_decode(base64_decode($config_serial), true) : "";

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $instance = grab_array_var($inargs, "instance", "");
            $port = grab_array_var($inargs, "port", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $instancename = grab_array_var($inargs, "instancename", "");
            $operation = grab_array_var($inargs, "operation", "");

            $hostname = nagiosccm_replace_user_macros($hostname);
            $address = nagiosccm_replace_user_macros($address);
            $port = nagiosccm_replace_user_macros($port);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            $use_2008_deprecated_stats = false;     # Tests that are deprecated (not available) after 2008
            $use_2008_plus_stats = true;            # Tests available in 2008 and newer.

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($hostname) == false)
                $errmsg[$errors++] = "<b>"._("Host Name")."</b>"._(" is required.");
            if (have_value($address) == false)
                $errmsg[$errors++] = "IP/Host <b>"._("Address")."</b> of the Database Server"._(" is required.");
            if (have_value($mssql_version) == false)
                $errmsg[$errors++] = "<b>"._("MSSQL Version")."</b>"._(" is required.");
            if (have_value($port) == false && have_value($instance) == false)
                $errmsg[$errors++] = _("Either ")."<b>"._("Port")."</b>"._(" number or ")."<b>"._("Database Server Instance")."</b>"._(" name is required.");
            if (have_value($username) == false)
                $errmsg[$errors++] = "<b>"._("Username")."</b>"._(" is required.");
            if (have_value($password) == false && $operation != 'update')   # Update preserves the existing password, unless overridden.
                $errmsg[$errors++] = "<b>"._("Password")."</b>"._(" is required.");
            if (have_value($instancename) == false) # This is hardcoded, above.
                $errmsg[$errors++] = "<b>"._("Database")."</b>"._(" name is required.  This may be a system or user defined database.");
            if ($port && $instance) {
                // instance overrides port
                $inargs["port"] = "";   // This is also handled in the plugin.
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $hostname = grab_array_var($inargs, "hostname");
            $address = grab_array_var($inargs, "ip_address");
            $port = grab_array_var($inargs, "port", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $instance = grab_array_var($inargs, "instance", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $instancename = grab_array_var($inargs, "instancename", "master");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "");

            # Only set if previously on step 2, then "back" to step 1, then "next" to step 2.
            # Set as hidden in step1.php
            $services_serial = grab_array_var($inargs, "services_serial");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");
            $config_serial = grab_array_var($inargs, "config_serial");

            $use_2008_deprecated_stats = false;
            $use_2008_plus_stats = true;

            switch ($mssql_version) {

                // Log Cache Hitrate should work for these versions (2008 R2 has been verified)
                case "2000":
                case "2000-64":
                case "2005":
                    $use_2008_plus_stats = false;       # Works with 2008+
                case "2008":
                case "2008-R2":
                    $use_2008_deprecated_stats = true;  # Works with 2008 or lower
                    break;
            }

            // Load the data for the selected host/node, unless we already have data.
            if (!empty($selectedhostconfig) && empty($services_serial) && empty($serviceargs_serial)) {
                $config = get_configwizard_config($wizard_name, $selectedhostconfig);

                #$hostname = grab_array_var($inargs, "hostname", $ha);
                $hostname = ($operation == 'copy') ? $hostname : $config['hostname'];
                $services = $config['services'];
                $serviceargs = $config['serviceargs'];

                ###
                # Bug fix:
                #   This only affects existing configs.  Future and modified configs will be correct.
                #   Custom checks should be split between services (on/off) and serviceargs (data).
                #
                if (isset($services['process']) && !isset($serviceargs['process'])) {
                    $custom_metrics = $services['process'];

                    foreach ($custom_metrics as $idx => $custom_metric) {
                        foreach ($custom_metric as $key => $value) {
                            if ($key == 'monitor') {
                                $services['process'][$idx] = $value;
                            } else {
                                $serviceargs['process'][$idx][$key] = $value;
                            }
                        }
                    }
                }

                $config['services']['process'] = (array_key_exists('process', $services) ? $services['process'] : '');
                $config['serviceargs']['process'] = $serviceargs['process'];

                #
                # End Bug fix
                ###

            // Load data from prev/next iterations.
            } else if ($services_serial != '' && $serviceargs_serial != '') {
                $services = json_decode(base64_decode($services_serial), true);
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
                $config = (!empty($config_serial)) ? json_decode(base64_decode($config_serial), true) : "";

            // Load default metrics.
            } else {
                $services = grab_array_var($inargs, "services", array_merge(
                    array(
                        "activetransactions" => "on",
                        "connectiontime" => "on",
                        "datafilesize" => "on",
                        "logbytesflushed" => "on",  # new
                        "logfilesize" => "on",      # new
                        "logfilesusedsize" => "on", # new
                        "logfileusage" => "on",
                        "logflushes" => "on",
                        "logflushwaits" => "on",    # new
                        "loggrowths" => "on",
                        "logshrinks" => "on",
                        "logtrunct" => "on",
                        "logwait" => "on",
                        "transactionspersec" => "on",
                        "transactionspersec" => "on",
                        "monitor" => "off",
                    ),
                    (($use_2008_deprecated_stats) ? array("logcachehitrate" => "on") : array())
                ));

                $serviceargs = grab_array_var($inargs, "serviceargs", array_merge(
                    array(
                        "activetransactions_warning" => "10",       # standard
                        "activetransactions_critical" => "20",
                        "connectiontime_warning" => "1",            # standard (time2connect)
                        "connectiontime_critical" => "5",
                        "datafilesize_warning" => "10000",          # standard
                        "datafilesize_critical" => "100000",
                        "logbytesflushed_warning" => "10000",       # delta - new
                        "logbytesflushed_critical" => "50000",
                        "logfilesize_warning" => "40000",           # standard - new
                        "logfilesize_critical" => "60000",
                        "logfilesusedsize_warning" => "4400",       # standard - new
                        "logfilesusedsize_critical" => "5000",
                        "logfileusage_warning" => array("0", "80"), # standard
                        "logfileusage_critical" => array("0", "90"),
                        "logflushes_warning" => "20",               # delta
                        "logflushes_critical" => "30",
                        "logflushwaits_warning" => "500",           # delta - new
                        "logflushwaits_critical" => "1000",
                        "loggrowths_warning" => "20",               # standard
                        "loggrowths_critical" => "30",
                        "logshrinks_warning" => "20",               # standard
                        "logshrinks_critical" => "30",
                        "logtrunct_warning" => "20",                # standard
                        "logtrunct_critical" => "30",
                        "logwait_warning" => "100",                 # delta
                        "logwait_critical" => "1000",
                        "transactionspersec_warning" => "10",       # delta
                        "transactionspersec_critical" => "20",
                    ),
                    (($use_2008_deprecated_stats) ?
                        array(
                            "logcachehitrate_warning" => array("30", "50"),  # ratio
                            "logcachehitrate_critical" => array("0", "30")
                        ) : array()
                    )
                ));
            }

            // Custom Service Checks.
            // Create only one default custom process to monitor... we will add more via JS if someone wants to add more
            $default_custom_services = array();
            $default_custom_serviceargs['process'][0]['monitor'] = 'off';
            $default_custom_serviceargs['process'][0]['counter_name'] = '';
            $default_custom_serviceargs['process'][0]['display_name'] = '';
            $default_custom_serviceargs['process'][0]['unit'] = '';
            $default_custom_serviceargs['process'][0]['modifier'] = '';
            $default_custom_serviceargs['process'][0]['ring_buffer_type'] = '';
            $default_custom_serviceargs['process'][0]['xpath'] = '';
            $default_custom_serviceargs['process'][0]['warning'] = 60;
            $default_custom_serviceargs['process'][0]['critical'] = 100;


            if (!isset($services['process'])) {
                $services = array_merge($services, $default_custom_services);
                $serviceargs = array_merge($serviceargs, $default_custom_serviceargs);
            }

            $select_options = array_merge(
                [
                    "Active Transactions" => "activetransactions",
                    "Connection Time" => "connectiontime",
                    "Database Size" => "datafilesize",
                    "Log File Usage" => "logfileusage",
                    "Log Bytes Flushed/sec" => "logbytesflushed",
                    "Log File(s) Size (KB)" => "logfilesize",
                    "Log File(s) Used Size (KB)" => "logfilesusedsize",
                    "Log Flush Waits/sec" => "logflushwaits",
                    "Log Flushes Per Second" => "logflushes",
                    "Log Growths" => "loggrowths",
                    "Log Truncations" => "logtrunct",
                    "Log Waits" => "logwait",
                    "Transactions Per Second" => "transactionspersec",
                ],
                (($use_2008_deprecated_stats) ?
                    [
                    "Log Cache Hitrate" => "logcachehitrate",
                    ] : []
                )
            );

            $service_tooltips = [
                "activetransactions" => "Monitor the number of active transactions",
                "connectiontime" => "Monitor the time it takes to connect to the database",
                "datafilesize" => "Monitor the size of the database. Value is in KB.",
                "logcachehitrate" => "Monitor the log cache hit rate",
                "logfileusage" => "Monitor how much of the Log File is in use",
                "logbytesflushed" => "Monitor the log bytes flushed per second",
                "logfilesize" => "Monitor the log files size in KB",
                "logfilesusedsize" => "Monitor the log files used size in KB",
                "logflushwaits" => "Monitor the log flush waits per second",
                "logflushes" => "Monitor the log flush rate per second",
                "loggrowths" => "Monitor the log growths due to improperly sized partitions",
                "logtrunct" => "Monitor the log truncations due to malformed tables",
                "logwait" => "Monitor the log waits due to small log buffers",
                "transactionspersec" => "Monitor the transactions per second",
            ];

            // Build Multi select HTML
            $service_list = '';
            $services_array = [];
            foreach ($select_options as $name => $service) {
                $name = encode_form_val($name);
                $safe_name = $service;
                $services_array[$name] = $service;
                $selected = '';
                if (isset($services[$service]) && $services[$service] == 'on'){ $selected = ' selected'; }
                $service_list .= "<option value='$safe_name' $selected>$name</option>";
            }

            $service_select_html = '<select name="services-select[]" id="services-select" multiple  class="form form-control metrics-select multiselect form-select">';
            $service_select_html .= $service_list;
            $service_select_html .= '</select>';

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $instance = grab_array_var($inargs, "instance", "");
            $port = grab_array_var($inargs, "port", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $instancename = grab_array_var($inargs, "instancename", "");
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $instance = grab_array_var($inargs, "instance", "");
            $port = grab_array_var($inargs, "port", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $instancename = grab_array_var($inargs, "instancename", "");
            $operation = grab_array_var($inargs, "operation", "");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $config_serial = grab_array_var($inargs, "config_serial", "");

            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $services_serial = (!empty($services)) ? base64_encode(json_encode($services)) : grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = (!empty($serviceargs)) ? base64_encode(json_encode($serviceargs)) : grab_array_var($inargs, "serviceargs_serial", "");

            ob_start();
?>
        <input type="hidden" id="ip_address" name="ip_address" value="<?= encode_form_val($address) ?>"/>
        <input type="hidden" id="hostname" name="hostname" value="<?= encode_form_val($hostname) ?>"/>
        <input type="hidden" id="port" name="port" value="<?= encode_form_val($port) ?>"/>
        <input type="hidden" id="mssql_version" name="mssql_version" value="<?= encode_form_val($mssql_version) ?>"/>
        <input type="hidden" id="tds_version" name="tds_version" value="<?= encode_form_val($tds_version) ?>"/>
        <input type="hidden" id="instance" name="instance" value="<?= encode_form_val($instance) ?>"/>
        <input type="hidden" id="username" name="username" value="<?= encode_form_val($username) ?>"/>
        <input type="hidden" id="password" name="password" value="<?= encode_form_val($password) ?>"/>
        <input type="hidden" id="instancename" name="instancename" value="<?= encode_form_val($instancename) ?>"/>
        <input type="hidden" id="services_serial" name="services_serial" value="<?= encode_form_val($services_serial) ?>"/>
        <input type="hidden" id="serviceargs_serial" name="serviceargs_serial" value="<?= encode_form_val($serviceargs_serial) ?>"/>
        <input type="hidden" id="operation" name="operation" value="<?= encode_form_val($operation) ?>"/>
        <input type="hidden" id="selectedhostconfig" name="selectedhostconfig" value="<?= encode_form_val($selectedhostconfig) ?>"/>
        <input type="hidden" id="config_serial" name="config_serial" value="<?= encode_form_val($config_serial) ?>"/>
<?php
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '
            ';

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $hostaddress = $address;
            $instance = grab_array_var($inargs, "instance", "");
            $port = grab_array_var($inargs, "port", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $instancename = grab_array_var($inargs, "instancename", "");
            $operation = grab_array_var($inargs, "operation", "");

            // Escape special characters that might break the config, etc.
            $username = nagiosccm_replace_command_line($username);
            $password = nagiosccm_replace_command_line($password);

            // We're updating, so preserve the existing password.
            if (have_value($password) == false && $operation == 'update') {
                $password = get_configwizard_config_value($wizard_name, $hostname, 'password');
            }

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["port"] = $port;
            $meta_arr["instance"] = $instance;
            $meta_arr["mssql_version"] = $mssql_version;
            $meta_arr["tds_version"] = $tds_version;
            $meta_arr["username"] = $username;
            $meta_arr["password"] = $password;
            $meta_arr["instancename"] = $instancename;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            // NOTE: This wizard now uses the check_mssql_server.php plugin (under the covers).
            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_mssqldatabase_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "mssqldatabase.png",
                    "statusmap_image" => "mssqldatabase.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            $perftype = 'default';

            switch ($mssql_version) {
                case "AZURESQLDB":
                    $perftype = 'azuresqldb';
                    break;

                case "PDW":
                    $perftype = 'pdw';
                    break;

                case "SQLDW":
                    $perftype = 'sqldw';
                    break;

                case "2005":
                case " 2000":
                    $perftype = 'deprecated';
                    break;
            }

            // common plugin opts
            // checktype = "database", because this is the "database" wizard.
            $commonopts = " --checktype 'database' -U '$username' -P '$password' --instancename '$instancename' --perftype $perftype ";
            if ($tds_version) {
                $commonopts .= "--tdsversion $tds_version ";
            }

            $instancetext = '';

            if ($instance) {
                $commonopts .= "-I '$instance' ";
                $instancetext = " - $instance";
            }

            if ($port) {
                $commonopts .= "-p $port ";
            }

            foreach ($services as $service => $args) {

                $pluginopts = "";
                $pluginopts .= $commonopts;

                switch ($service) {

                    case "connectiontime":

                        $pluginopts .= "--mode time2connect --warning ".$serviceargs["connectiontime_warning"]." --critical ".$serviceargs["connectiontime_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Connection Time".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logfileusage":

                        $pluginopts .= "--mode logfileusage --warning ".$serviceargs["logfileusage_warning"][0].":".$serviceargs["logfileusage_warning"][1]." --critical ".$serviceargs["logfileusage_critical"][0].":".$serviceargs["logfileusage_critical"][1];

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log File Usage".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "datafilesize":

                        $pluginopts .= "--mode datasize --warning ".$serviceargs["datafilesize_warning"]." --critical ".$serviceargs["datafilesize_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Database Size".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "activetransactions":

                        $pluginopts .= "--mode activetrans --warning ".$serviceargs["activetransactions_warning"]." --critical ".$serviceargs["activetransactions_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Active Transactions".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logcachehitrate":

                        $pluginopts .= "--mode logcachehit --warning ".$serviceargs["logcachehitrate_warning"][0].":".$serviceargs["logcachehitrate_warning"][1]." --critical ".$serviceargs["logcachehitrate_critical"][0].":".$serviceargs["logcachehitrate_critical"][1];

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log Cache Hit Rate".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;


                    case "logwait":

                        $pluginopts .= "--mode logwait --warning ".$serviceargs["logwait_warning"]." --critical ".$serviceargs["logwait_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log Flush Wait Time".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "loggrowths":

                        $pluginopts .= "--mode loggrowths --warning ".$serviceargs["loggrowths_warning"]." --critical ".$serviceargs["loggrowths_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log Growths".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logshrinks":

                        $pluginopts .= "--mode logshrinks --warning ".$serviceargs["logshrinks_warning"]." --critical ".$serviceargs["logshrinks_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log Shrinks".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logtrunct":

                        $pluginopts .= "--mode logtruncs --warning ".$serviceargs["logtrunct_warning"]." --critical ".$serviceargs["logtrunct_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log Truncations".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logbytesflushed":

                        $pluginopts .= "--mode logbytesflushed --warning ".$serviceargs["logbytesflushed_warning"]." --critical ".$serviceargs["logbytesflushed_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log Bytes Flushed / Sec".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logfilesize":

                        $pluginopts .= "--mode logfilessize --warning ".$serviceargs["logfilesize_warning"]." --critical ".$serviceargs["logfilesize_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log Files Size".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logfilesusedsize":

                        $pluginopts .= "--mode logfilesused --warning ".$serviceargs["logfilesusedsize_warning"]." --critical ".$serviceargs["logfilesusedsize_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log Files Used Size".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logflushwaits":

                        $pluginopts .= "--mode logflushwaits --warning ".$serviceargs["logflushwaits_warning"]." --critical ".$serviceargs["logflushwaits_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log Flush Waits / Sec".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logflushes":

                        $pluginopts .= "--mode logflushes --warning ".$serviceargs["logflushes_warning"]." --critical ".$serviceargs["logflushes_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Log Flushes / Sec".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "transactionspersec":

                        $pluginopts .= "--mode transpsec --warning ".$serviceargs["transactionspersec_warning"]." --critical ".$serviceargs["transactionspersec_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => $instancename." MSSQL Transactions / Sec".$instancetext,
                            "use" => "xiwizard_mssqldatabase_service",
                            "check_command" => "check_xi_mssql_database2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "process":
                        foreach ($args as $i => $servicearg) {
                            if (!array_key_exists('process', $service) || empty($servicearg['counter_name'])) {
                                continue;
                            } else if ($service['process'][$i] == 'off') {
                                continue;
                            }

                            $pluginCustomOpts  = '--mode \'custom\' --warning '.$servicearg["warning"].' --critical '.$servicearg["critical"].' ';
                            $pluginCustomOpts .= '--custom \'("counter_name" : "'.$servicearg["counter_name"].'", "instance_name" : "'.$instancename.'", "unit" : "'.$servicearg["unit"].'", "modifier" : "'.$servicearg["modifier"].'", "ring_buffer_type" : "'.$servicearg["ring_buffer_type"].'", "xpath" : "'.$servicearg["xpath"].'")\'';

                            $serviceDescription = $servicearg['display_name'];

                            if (empty($servicearg['display_name'])) {
                                # e.g. Cache Hit Ratio to 'cachehitratio'
                                $serviceDescription = ucwords(str_replace("%", "Pct", $servicearg["counter_name"]));
                                $serviceDescription = (!empty($instancename) ? $instancename." " : "")."MSSQL ".$serviceDescription;
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $serviceDescription,
                                "use" => "xiwizard_mssqlserver_service",
                                "_xiwizard" => $wizard_name,
                                "check_command" => "check_xi_mssql_database2!".$pluginopts.$pluginCustomOpts,
                            );
                        }

                        break;

                    default:

                        break;
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:

            break;
    }

    return $output;
}

?>
