<?php
//
// MSSQL Server Config Wizard
// Copyright (c) 2010-2025 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__).'/../configwizardhelper.inc.php');

mssqlserver_configwizard_init();

function mssqlserver_configwizard_init()
{
    $name = "mssql_server";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "3.2.0",
        CONFIGWIZARD_IS_PREMIUM => true,
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a MSSQL Server"),
        CONFIGWIZARD_DISPLAYTITLE => _("MSSQL Server"),
        CONFIGWIZARD_FUNCTION => "mssqlserver_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "mssqlserver.png",
        CONFIGWIZARD_FILTER_GROUPS => array('windows', 'database', 'server'),
        CONFIGWIZARD_REQUIRES_VERSION => 70000
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function mssqlserver_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    global $configwizards;
    $wizard_name = "mssql_server";
    $wizard_title = $configwizards[$wizard_name]['display_title'];

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "1433");
            $instance = grab_array_var($inargs, "instance", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $instancename = grab_array_var($inargs, "instancename", "");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            # These will only be set by going back from step 2.
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");
            $config_serial = grab_array_var($inargs, "config_serial");

            $config = (!empty($config_serial)) ? json_decode(base64_decode($config_serial), true) : "";

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $instance = grab_array_var($inargs, "instance", "");
            $port = grab_array_var($inargs, "port", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $instancename = grab_array_var($inargs, "instancename", "");
            $operation = grab_array_var($inargs, "operation", "");

            $hostname = nagiosccm_replace_user_macros($hostname);
            $address = nagiosccm_replace_user_macros($address);
            $port = nagiosccm_replace_user_macros($port);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            $use_2008_deprecated_stats = false;     # Tests that are deprecated (not available) after 2008
            $use_2008_plus_stats = true;            # Tests available in 2008 and newer.
            $use_2014_plus_stats = true;            # Tests available in 2014 and newer.

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($hostname) == false)
                $errmsg[$errors++] = "<b>"._("Host Name")."</b>"._(" is required.");
            if (have_value($address) == false)
                $errmsg[$errors++] = "IP/Host <b>"._("Address")."</b> of the Database Server"._(" is required.");
            if (have_value($mssql_version) == false)
                $errmsg[$errors++] = "<b>"._("MSSQL Version")."</b>"._(" is required.");
            if (have_value($port) == false && have_value($instance) == false)
                $errmsg[$errors++] = _("Either ")."<b>"._("Port")."</b>"._(" number or ")."<b>"._("Database Server Instance")."</b>"._(" name is required.");
            if (have_value($username) == false)
                $errmsg[$errors++] = "<b>"._("Username")."</b>"._(" is required.");
            if (have_value($password) == false && $operation != 'update')   # Update preserves the existing password, unless overridden.
                $errmsg[$errors++] = "<b>"._("Password")."</b>"._(" is required.");
            if ($port && $instance) {
                // instance overrides port
                $inargs["port"] = "";   // This is also handled in the plugin.
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $hostname = grab_array_var($inargs, "hostname");
            $address = grab_array_var($inargs, "ip_address");
            $port = grab_array_var($inargs, "port", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $instance = grab_array_var($inargs, "instance", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $instancename = grab_array_var($inargs, "instancename", "");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "");

            # Only set if previously on step 2, then "back" to step 1, then "next" to step 2.
            # Set as hidden in step1.php
            $services_serial = grab_array_var($inargs, "services_serial");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");
            $config_serial = grab_array_var($inargs, "config_serial");

            $use_2008_deprecated_stats = false;
            $use_2008_plus_stats = true;
            $use_2014_plus_stats = true;

            switch ($mssql_version) {

                // Free Pages, Stolen Pages and Total Pages, should work for these versions (2008 R2 has been verified)
                case "2000":
                case "2000-64":
                case "2005":
                    $use_2008_plus_stats = false;       # Works with 2008+
                case "2008":
                case "2008-R2":
                    $use_2008_deprecated_stats = true;  # Works with 2008 or lower
                case "2012":
                    $use_2014_plus_stats = false;       # Works with 2014+
                    break;
            }

            // Load the data for the selected host/node, unless we already have data.
            if (!empty($selectedhostconfig) && empty($services_serial) && empty($serviceargs_serial)) {
                $config = get_configwizard_config($wizard_name, $selectedhostconfig);

                #$hostname = grab_array_var($inargs, "hostname", $ha);
                $hostname = ($operation == 'copy') ? $hostname : $config['hostname'];
                $services = $config['services'];
                $serviceargs = $config['serviceargs'];

                ###
                # Bug fix:
                #   This only affects existing configs.  Future and modified configs will be correct.
                #   Custom checks should be split between services (on/off) and serviceargs (data).
                #
                if (isset($services['process']) && !isset($serviceargs['process'])) {
#echo("BUG??<br>");
                    $custom_metrics = $services['process'];

                    foreach ($custom_metrics as $idx => $custom_metric) {
                        foreach ($custom_metric as $key => $value) {
                            if ($key == 'monitor') {
                                $services['process'][$idx] = $value;
                            } else {
                                $serviceargs['process'][$idx][$key] = $value;
                            }
                        }
                    }
                }

                $config['services']['process'] = $services['process'];
                $config['serviceargs']['process'] = $serviceargs['process'];

                #
                # End Bug fix
                ###

            // Load data from prev/next iterations.
            } else if ($services_serial != '' && $serviceargs_serial != '') {
                $services = json_decode(base64_decode($services_serial), true);
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
                $config = (!empty($config_serial)) ? json_decode(base64_decode($config_serial), true) : "";

            // Load default metrics.
            } else {
                $services = grab_array_var($inargs, "services", array_merge(
                    array(
                        "autoparamattempts" => "on",    # new
                        "avglatchwait" => "on",         # new
                        "averagewait" => "on",
                        "batchreq" => "on",             # missing from the wizard
                        "bufferhitratio" => "on",
                        "cachehit" => "on",             # missing from the wizard
                        "cacheobjcounts" => "on",       # new
                        "cacheobjsinuse" => "on",       # new
                        "cachepages" => "on",           # new
                        "checkpoints" => "on",
                        "connectiontime" => "on",
                        "connections" => "on",          # missing from the wizard
                        "cpu" => "on",                  # missing from the wizard
                        "databasepages" => "on",
                        "deadlocks" => "on",
                        "failedautoparams" => "on",     # new
                        "forwardedrecords" => "on",     # new
                        "freeliststalls" => "on",       # new
                        "fullscans" => "on",            # missing from the wizard
                        "grantedwsmem" => "on",         # new
                        "indexsearches" => "on",        # new
                        "latchwaits" => "on",           # new
                        "lazywrites" => "on",
                        "lockrequests" => "on",
                        "locktimeouts" => "on",
                        "lockwait" => "on",
                        "lockwaits" => "on",
                        "logins" => "on",               # new
                        "logouts" => "on",              # new
                        "longesttrans" => "on",         # new
                        "maxwsmem" => "on",             # new
                        "memgrantsoutstand" => "on",    # new
                        "memgrantspend" => "on",        # new
                        "numsuperlatches" => "on",      # new
                        "pagelife" => "on",             # missing from the wizard
                        "pagelooks" => "on",
                        "pagereads" => "on",
                        "pagesplits" => "on",
                        "pagewrites" => "on",
                        "processesblocked" => "on",     # new
                        "readahead" => "on",
                        "safeautoparams" => "on",       # new
                        "sqlattentionrate" => "on",     # new
                        "sqlcompilations" => "on",      # missing from the wizard
                        "sqlrecompilations" => "on",    # new
                        "superlatchdemotes" => "on",    # new
                        "superlatchpromotes" => "on",   # new
                        "tablelockescalate" => "on",    # new
                        "targetpages" => "on",
                        "targetsrvmem" => "on",         # new
                        "totallatchwait" => "on",       # new
                        "totalsrvmem" => "on",          # new
                        "unsafeautoparams" => "on",     # new
                        "usercons" => "on",             # new
                        "workfilescreated" => "on",     # new
                        "worktablescreated" => "on",    # new
                    ),
                    (($use_2008_plus_stats) ?
                        array(
                            "connectionreset" => "on",  # new
                            "memory" => "on",           # missing from the wizard
                        ) : array()
                    ),
                    (($use_2008_deprecated_stats) ?
                        array(
                            "freepages" => "on",
                            "stolenpages" => "on",
                            "totalpages" => "on",       # missing from the wizard
                        ) : array()
                    ),
                    (($use_2014_plus_stats) ?
                        array(
                            "worktablescacheratio" => "on", # new
                        ) : array()
                    )
                ));

                $serviceargs = grab_array_var($inargs, "serviceargs", array_merge(
                    array(
                        "autoparamattempts_warning" => "100",   # new - delta
                        "autoparamattempts_critical" => "200",  # new - delta
                        "avglatchwait_warning" => "100",        # new - delta ratio
                        "avglatchwait_critical" => "200",       # new - delta ratio
                        "averagewait_warning" => "20",
                        "averagewait_critical" => "30",
                        "batchreq_warning" => "20",         # missing from the wizard
                        "batchreq_critical" => "30",        # missing from the wizard
                        "bufferhitratio_warning" => "90",
                        "bufferhitratio_critical" => "95",
                        "cachehit_warning" => "20",         # missing from the wizard
                        "cachehit_critical" => "30",        # missing from the wizard
                        "cacheobjcounts_warning" => "10",   # new - standard (point in time)
                        "cacheobjcounts_critical" => "20",  # new - standard (point in time)
                        "cacheobjsinuse_warning" => "10",   # new - standard (point in time)
                        "cacheobjsinuse_critical" => "20",  # new - standard (point in time)
                        "cachepages_warning" => "500",      # new - standard (point in time)
                        "cachepages_critical" => "1000",    # new - standard (point in time)
                        "checkpoints_warning" => "20",
                        "checkpoints_critical" => "30",
                        "connectiontime_warning" => "1",
                        "connectiontime_critical" => "5",
                        "connections_warning" => "20",      # missing from the wizard
                        "connections_critical" => "30",     # missing from the wizard
                        "cpu_warning" => "20",              # missing from the wizard
                        "cpu_critical" => "30",             # missing from the wizard
                        "databasepages_warning" => "300",
                        "databasepages_critical" => "600",
                        "deadlocks_warning" => "20",
                        "deadlocks_critical" => "30",
                        "failedautoparams_warning" => "50",     # new - delta
                        "failedautoparams_critical" => "100",   # new - delta
                        "forwardedrecords_warning" => "50",     # new - delta
                        "forwardedrecords_critical" => "100",   # new - delta
                        "freeliststalls_warning" => "2",        # new - delta
                        "freeliststalls_critical" => "4",       # new - delta
                        "fullscans_warning" => "20",        # missing from the wizard
                        "fullscans_critical" => "30",       # missing from the wizard
                        "grantedwsmem_warning" => "10",         # new - standard (point in time)
                        "grantedwsmem_critical" => "20",        # new - standard (point in time)
                        "indexsearches_warning" => "5000",      # new - delta
                        "indexsearches_critical" => "10000",    # new - delta
                        "latchwaits_warning" => "1000",         # new - delta
                        "latchwaits_critical" => "5000",        # new - delta
                        "lazywrites_warning" => "20",
                        "lazywrites_critical" => "30",
                        "lockrequests_warning" => "20",
                        "lockrequests_critical" => "30",
                        "locktimeouts_warning" => "20",
                        "locktimeouts_critical" => "30",
                        "lockwaits_warning" => "20",            # Ideally 0 or close to 0
                        "lockwaits_critical" => "30",
                        "lockwait_warning" => "2000",
                        "lockwait_critical" => "3000",
                        "logins_warning" => "2",                # new - delta
                        "logins_critical" => "4",               # new - delta
                        "logouts_warning" => "2",               # new - delta
                        "logouts_critical" => "4",              # new - delta
                        "longesttrans_warning" => "5",          # new - standard (seconds)
                        "longesttrans_critical" => "10",        # new - standard (seconds)
                        "maxwsmem_warning" => "1600000",        # new - standard
                        "maxwsmem_critical" => "2000000",       # new - standard
                        "memgrantsoutstand_warning" => "0",     # new - standard
                        "memgrantsoutstand_critical" => "1",    # new - standard
                        "memgrantspend_warning" => "0",         # new - standard
                        "memgrantspend_critical" => "1",        # new - standard
                        "numsuperlatches_warning" => "10",      # new - standard
                        "numsuperlatches_critical" => "20",     # new - standard
                        "pagelife_warning" => "20",         # missing from the wizard
                        "pagelife_critical" => "30",        # missing from the wizard
                        "pagelooks_warning" => "10",
                        "pagelooks_critical" => "20",
                        "pagereads_warning" => "20",
                        "pagereads_critical" => "30",
                        "pagesplits_warning" => "20",           # Ideally < 20% of batch requests/sec
                        "pagesplits_critical" => "30",
                        "pagewrites_warning" => "20",
                        "pagewrites_critical" => "30",
                        "processesblocked_warning" => "0",      # new - standard - Ideally 0 (nothing blocked)
                        "processesblocked_critical" => "3",     # new - standard
                        "readahead_warning" => "40",
                        "readahead_critical" => "50",
                        "safeautoparams_warning" => "20",       # new - delta - ???
                        "safeautoparams_critical" => "50",      # new - delta
                        "sqlattentionrate_warning" => "1",      # new - delta - lower the better
                        "sqlattentionrate_critical" => "5",     # new - delta
                        "sqlcompilations_warning" => "20",  # missing from the wizard - ideally 1 compile/10 batch requests.
                        "sqlcompilations_critical" => "30", # missing from the wizard
                        "sqlrecompilations_warning" => "5",     # new - delta - ideally less than 10% of compilations/sec
                        "sqlrecompilations_critical" => "10",   # new - delta
                        "superlatchdemotes_warning" => "5",     # new - delta
                        "superlatchdemotes_critical" => "10",   # new - delta
                        "superlatchpromotes_warning" => "5",    # new - delta
                        "superlatchpromotes_critical" => "10",  # new - delta
                        "tablelockescalate_warning" => "5",     # new - delta
                        "tablelockescalate_critical" => "10",   # new - delta
                        "targetpages_warning" => "70000",
                        "targetpages_critical" => "90000",
                        "targetsrvmem_warning" => "1900000",    # new - standard
                        "targetsrvmem_critical" => "2000000",   # new - standard
                        "totallatchwait_warning" => "5",        # new - delta
                        "totallatchwait_critical" => "10",      # new - delta
                        "totalsrvmem_warning" => "1500000",     # new - standard
                        "totalsrvmem_critical" => "1200000",    # new - standard
                        "unsafeautoparams_warning" => "5",      # new - delta - lower is better
                        "unsafeautoparams_critical" => "10",    # new - delta
                        "usercons_warning" => "20",             # new - standard
                        "usercons_critical" => "50",            # new - standard
                        "workfilescreated_warning" => "20",     # new - delta
                        "workfilescreated_critical" => "30",    # new - delta
                        "worktablescreated_warning" => "20",    # new - delta
                        "worktablescreated_critical" => "30",   # new - delta
                    ),
                    (($use_2008_plus_stats) ?
                        array(
                            "connectionreset_warning" => "1000",    # new
                            "connectionreset_critical" => "2000",   # new
                            "memory_warning" => "20",   # missing from the wizard
                            "memory_critical" => "30",  # missing from the wizard
                        ) : array()
                    ),
                    (($use_2008_deprecated_stats) ?
                        array(
                            "freepages_warning" => "10",
                            "freepages_critical" => "20",
                            "stolenpages_warning" => "500",
                            "stolenpages_critical" => "700",
                            "totalpages_warning" => "500",  # missing from the wizard
                            "totalpages_critical" => "700", # missing from the wizard
                        ) : array()
                    ),
                    (($use_2014_plus_stats) ?
                        array(
                            "worktablescacheratio_warning" => "95:",
                            "worktablescacheratio_critical" => "90:",
                        ) : array()
                    )
                ));
            }

            $select_options = array_merge(
                array(
                    "Auto-Param Attempts/sec" => "autoparamattempts",
                    "Average Latch Wait Time (ms)" => "avglatchwait",
                    "Average Wait Time" => "averagewait",
                    "Batch Requests" => "batchreq",
                    "Buffer Hit Ratio" => "bufferhitratio",
                    "Cache Hit Ratio" => "cachehit",
                    "Cache Object Counts" => "cacheobjcounts",
                    "Cache Objects in use" => "cacheobjsinuse",
                    "Cache Pages" => "cachepages",
                    "Checkpoint Pages/sec" => "checkpoints",
                    "Connection Time" => "connectiontime",
                    "Connections" => "connections",
                    "CPU Utilization" => "cpu",
                    "Database Pages" => "databasepages",
                    "Deadlocks" => "deadlocks",
                    "Failed Auto-Params/sec" => "failedautoparams",
                    "Forwarded Records/sec" => "forwardedrecords",
                    "Free List Stalls" => "freeliststalls",
                    "Full Scans" => "fullscans",
                    "Granted Workspace Memory" => "grantedwsmem",
                    "Index Searches/sec" => "indexsearches",
                    "Latch Waits/sec" => "latchwaits",
                    "Lazy Writes" => "lazywrites",
                    "Lock Requests" => "lockrequests",
                    "Lock Timeouts" => "locktimeouts",
                    "Lock Wait Time" => "lockwait",
                    "Lock Waits" => "lockwaits",
                    "Logins/sec" => "logins",
                    "Logouts/sec" => "logouts",
                    "Longest Transaction Running Time" => "longesttrans",
                    "Maximum Workspace Memory (KB)" => "maxwsmem",
                    "Memory Grants Outstanding" => "memgrantsoutstand",
                    "Memory Grants Pending" => "memgrantspend",
                    "Number of SuperLatches" => "numsuperlatches",
                    "Page life expectancy" => "pagelife",
                    "Page Looks" => "pagelooks",
                    "Page Reads" => "pagereads",
                    "Page Splits" => "pagesplits",
                    "Page Writes" => "pagewrites",
                    "Processes Blocked" => "processesblocked",
                    "Read Aheads" => "readahead",
                    "Safe Auto-Params/sec" => "safeautoparams",
                    "SQL Attention Rate/sec" => "sqlattentionrate",
                    "SQL Compilations" => "sqlcompilations",
                    "SQL Re-Compilations/sec" => "sqlrecompilations",
                    "SuperLatch Demotions/sec" => "superlatchdemotes",
                    "SuperLatch Promotions/sec" => "superlatchpromotes",
                    "Table Lock Escalations/sec" => "tablelockescalate",
                    "Target Pages" => "targetpages",
                    "Target Server Memory (KB)" => "targetsrvmem",
                    "Total Latch Wait Time (ms)" => "totallatchwait",
                    "Total Server Memory (KB)" => "totalsrvmem",
                    "Unsafe Auto-Params/sec" => "unsafeautoparams",
                    "User Connections" => "usercons",
                    "Workfiles Created/sec" => "workfilescreated",
                    "Worktables Created/sec" => "worktablescreated",
                                ),
                (($use_2008_plus_stats) ?
                    array(
                        "Connection reset/sec" => "connectionreset",
                        "Memory" => "memory",
                    ) : array()
                ),
                (($use_2008_deprecated_stats) ?
                    array(
                        "Free Pages" => "freepages",
                        "Stolen Pages" => "stolenpages",
                        "Total Pages" => "totalpages",
                    ) : array()
                ),
                (($use_2014_plus_stats) ?
                    array(
                        "Worktables From Cache Ratio" => "worktablescacheratio",
                    ) : array()
                )
            );

            $service_tooltips = [
                "autoparamattempts" => "Monitor the number of auto-parameterization attempts per second.",
                "avglatchwait" => "Monitor the average time in milliseconds, for any latch requests that had to wait.",
                "averagewait" => "Monitor the average wait time for execution.",
                "batchreq" => "Monitor the number of batch requests/sec.",
                "bufferhitratio" => "Monitor the Buffer Hit Ratio.",
                "cachehit" => "Monitor the Cache Hit Ratio.",
                "cacheobjcounts" => "Monitor the number of objects in the cache.",
                "cacheobjsinuse" => "Monitor the number of cache objects in use.",
                "cachepages" => "Monitor the number of 8-kilobyte pages used by cache objects.",
                "checkpoints" => "The number of dirty pages per second.",
                "connectionreset" => "Total number of logins started from the connection pool (Delta).",
                "connectiontime" => "Monitor the time it takes to connect to the server.",
                "connections" => "Monitor the number of open connections to the server.",
                "cpu" => "Monitor the current CPU utilization.",
                "databasepages" => "The amount of database pages.",
                "deadlocks" => "The amount of deadlocks per sec.",
                "failedautoparams" => "Monitor the number of auto-parameterization attempts per second.",
                "forwardedrecords" => "Monitor the pointers created when rows have been moved to a new page in a heap.",
                "freeliststalls" => "Monitor the number of requests waiting for a free page/sec.",
                "freepages" => "Monitor the free pages.",
                "fullscans" => "The amount of full scans per sec.",
                "grantedwsmem" => "Monitor the total memory granted to executing processes.",
                "indexsearches" => "Monitor the number of index searches, within an index.",
                "latchwaits" => "Monitor the number of latches that had to wait, in the last second.",
                "lazywrites" => "The amount of lazy writes per sec.",
                "lockrequests" => "The amount of lock requests per sec.",
                "locktimeouts" => "The amount of lock timeouts per sec.",
                "lockwait" => "Monitor the lock wait time.",
                "lockwaits" => "The amount of lockwaits per sec.",
                "logins" => "Monitor the total number of logins started per second (not pooled connections).",
                "logouts" => "Monitor the total number of logout operations started per second.",
                "longesttrans" => "Monitor the time in seconds, of the longest running transaction.",
                "maxwsmem" => "Monitor the maximum amount of memory available for processes.",
                "memgrantsoutstand" => "Monitor the total number of processes per second successfully acquiring a workspace memory grant.",
                "memgrantspend" => "Monitor the total number of processes per second waiting for a workspace memory grant.",
                "memory" => "Percent of memory the server is using.",
                "numsuperlatches" => "Monitor the current number of SuperLatches.",
                "pagelife" => "Monitor the page life expectancy.",
                "pagelooks" => "Monitor the number of page looks per second.",
                "pagereads" => "The amount of page reads per sec.",
                "pagesplits" => "The amount of page splits per sec.",
                "pagewrites" => "The amount of page writes per sec.",
                "processesblocked" => "Monitor the number of currently blocked processes.",
                "readahead" => "The amount of readaheads per sec.",
                "safeautoparams" => "Monitor the number of safe auto-parameterization attempts per second.",
                "sqlattentionrate" => "Monitor the number of cancels and query timeouts occurring per second.",
                "sqlcompilations" => "Monitor the sql compilations per sec.",
                "sqlrecompilations" => "Monitor the number of statement recompiles per second.",
                "stolenpages" => "Monitor the amount of stolen pages.",
                "superlatchdemotes" => "Monitor the number of latches demoted from SuperLatches, in the last second.",
                "superlatchpromotes" => "Monitor the number of latches promoted to SuperLatches, in the last second.",
                "tablelockescalate" => "Monitor the number of times locks escalated from page or row to table-level.",
                "targetpages" => "The amount of target pages.",
                "targetsrvmem" => "Monitor the amount of memory SQL Server is using.",
                "totallatchwait" => "Monitor the total amount of time spent waiting for a latch, in the last second.",
                "totalpages" => "The total cumulative amount of pages.",
                "totalsrvmem" => "Monitor the amount of memory SQL Server is using.",
                "unsafeautoparams" => "Monitor the number of unsafe auto-parameterizations per second.",
                "usercons" => "Monitor the number of users currently connected.",
                "workfilescreated" => "Monitor the number of work files created per second.",
                "worktablescacheratio" => "Percentage of work tables created whose initial two pages were immediately available from the worktable cache.",
            ];

            // Custom Service Checks.
            // Create only one default process to monitor... we will add more via JS if someone wants to add more
            $default_custom_services = array();
            $default_custom_serviceargs['process'][0]['monitor'] = 'off';
            $default_custom_serviceargs['process'][0]['counter_name'] = '';
            $default_custom_serviceargs['process'][0]['display_name'] = '';
            $default_custom_serviceargs['process'][0]['instance_name'] = '';
            $default_custom_serviceargs['process'][0]['unit'] = '';
            $default_custom_serviceargs['process'][0]['modifier'] = '';
            $default_custom_serviceargs['process'][0]['ring_buffer_type'] = '';
            $default_custom_serviceargs['process'][0]['xpath'] = '';
            $default_custom_serviceargs['process'][0]['warning'] = 60;
            $default_custom_serviceargs['process'][0]['critical'] = 100;

            if (!isset($services['process'])) {
                $services = array_merge($services, $default_custom_services);
                $serviceargs = array_merge($serviceargs, $default_custom_serviceargs);
            }

            // Build Services Multi select HTML
            $service_list = '';
            $services_array = [];
            foreach ($select_options as $name => $service) {
                $name = encode_form_val($name);
                $safe_name = $service;
                $services_array[$name] = $service;
                $selected = '';
                if (isset($services[$service]) && $services[$service] == 'on'){ $selected = ' selected'; }
                $service_list .= "<option value='$safe_name' $selected>$name</option>";
            }

            $service_select_html = '<select name="services-select[]" id="services-select" multiple  class="form form-control metrics-select multiselect form-select">';
            $service_select_html .= $service_list;
            $service_select_html .= '</select>';


            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $instance = grab_array_var($inargs, "instance", "");
            $port = grab_array_var($inargs, "port", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $instance = grab_array_var($inargs, "instance", "");
            $port = grab_array_var($inargs, "port", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $operation = grab_array_var($inargs, "operation", "");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $config_serial = grab_array_var($inargs, "config_serial", "");

            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $services_serial = (!empty($services)) ? base64_encode(json_encode($services)) : grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = (!empty($serviceargs)) ? base64_encode(json_encode($serviceargs)) : grab_array_var($inargs, "serviceargs_serial", "");

            ob_start();
?>
        <input type="hidden" id="ip_address" name="ip_address" value="<?= encode_form_val($address) ?>"/>
        <input type="hidden" id="hostname" name="hostname" value="<?= encode_form_val($hostname) ?>"/>
        <input type="hidden" id="port" name="port" value="<?= encode_form_val($port) ?>"/>
        <input type="hidden" id="mssql_version" name="mssql_version" value="<?= encode_form_val($mssql_version) ?>">
        <input type="hidden" id="tds_version" name="tds_version" value="<?= encode_form_val($tds_version) ?>">
        <input type="hidden" id="instance" name="instance" value="<?= encode_form_val($instance) ?>"/>
        <input type="hidden" id="username" name="username" value="<?= encode_form_val($username) ?>"/>
        <input type="hidden" id="password" name="password" value="<?= encode_form_val($password) ?>"/>
        <input type="hidden" id="services_serial" name="services_serial" value="<?= encode_form_val($services_serial) ?>"/>
        <input type="hidden" id="serviceargs_serial" name="serviceargs_serial" value="<?= encode_form_val($serviceargs_serial) ?>"/>
        <input type="hidden" id="operation" name="operation" value="<?= encode_form_val($operation) ?>"/>
        <input type="hidden" id="selectedhostconfig" name="selectedhostconfig" value="<?= encode_form_val($selectedhostconfig) ?>"/>
        <input type="hidden" id="config_serial" name="config_serial" value="<?= encode_form_val($config_serial) ?>"/>
<?php
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '
            ';

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $hostaddress = $address;
            $instance = grab_array_var($inargs, "instance", "");
            $port = grab_array_var($inargs, "port", "");
            $mssql_version = grab_array_var($inargs, "mssql_version", "");
            $tds_version = grab_array_var($inargs, "tds_version", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $instancename = grab_array_var($inargs, "instancename", "");
            $operation = grab_array_var($inargs, "operation", "");

            // Escape special characters that might break the config, etc.
            $username = nagiosccm_replace_command_line($username);
            $password = nagiosccm_replace_command_line($password);

            // We're updating, so preserve the existing password.
#echo("Password [".$password."] operation [".$operation."] have_value(password) [".have_value($password)."]<p>");
            if (have_value($password) == false && $operation == 'update') {
                $password = get_configwizard_config_value($wizard_name, $hostname, 'password');
#echo("Original password [".$password."]<p>");
            }
#else echo("New password [".$password."]<p>");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["port"] = $port;
            $meta_arr["instance"] = $instance;
            $meta_arr["mssql_version"] = $mssql_version;
            $meta_arr["tds_version"] = $tds_version;
            $meta_arr["username"] = $username;
            $meta_arr["password"] = $password;
            $meta_arr["instancename"] = $instancename;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_mssqlserver_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "mssqlserver.png",
                    "statusmap_image" => "mssqlserver.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            $perftype = 'default';

            switch ($mssql_version) {
                case "AZURESQLDB":
                    $perftype = 'azuresqldb';
                    break;

                case "PDW":
                    $perftype = 'pdw';
                    break;

                case "SQLDW":
                    $perftype = 'sqldw';
                    break;

                case "2005":
                case " 2000":
                    $perftype = 'deprecated';
                    break;
            }

            // common plugin opts
            // checktype = "server", because this is the "server" wizard.
            $commonopts = " --checktype 'server' -U '$username' -P '$password' --perftype $perftype ";
            if ($tds_version) {
                $commonopts .= "--tdsversion $tds_version ";
            }

            $instancetext = '';

            if ($instance) {
                $commonopts .= "-I '$instance' ";
                $instancetext = " - $instance";
            }

            if ($port) {
                $commonopts .= "-p $port ";
            }

            foreach ($services as $service => $args) {

                $pluginopts = "";
                $pluginopts .= $commonopts;

                switch ($service) {

                    case "autoparamattempts":

                        $pluginopts .= "--mode autoparamattempts --warning ".$serviceargs["autoparamattempts_warning"]." --critical ".$serviceargs["autoparamattempts_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Auto-Param Attempts / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "avglatchwait":

                        $pluginopts .= "--mode avglatchwait --warning ".$serviceargs["avglatchwait_warning"]." --critical ".$serviceargs["avglatchwait_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Average Latch Wait Time".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "averagewait":

                        $pluginopts .= "--mode averagewait --warning ".$serviceargs["averagewait_warning"]." --critical ".$serviceargs["averagewait_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Average Wait Time".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "batchreq":

                        $pluginopts .= "--mode batchreq --warning ".$serviceargs["batchreq_warning"]." --critical ".$serviceargs["batchreq_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Batch Requests / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "bufferhitratio":

                        $pluginopts .= "--mode bufferhitratio --warning ".$serviceargs["bufferhitratio_warning"].": --critical ".$serviceargs["bufferhitratio_critical"].":";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Buffer Hit Ratio".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "cachehit":

                        $pluginopts .= "--mode cachehit --warning ".$serviceargs["cachehit_warning"].": --critical ".$serviceargs["cachehit_critical"].":";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Cache Hit Ratio".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "cacheobjcounts":

                        $pluginopts .= "--mode cacheobjcounts --warning ".$serviceargs["cacheobjcounts_warning"]." --critical ".$serviceargs["cacheobjcounts_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Cache Object Counts".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "cacheobjsinuse":

                        $pluginopts .= "--mode cacheobjsinuse --warning ".$serviceargs["cacheobjsinuse_warning"]." --critical ".$serviceargs["cacheobjsinuse_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Cache Objects in use".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "cachepages":

                        $pluginopts .= "--mode cachepages --warning ".$serviceargs["cachepages_warning"]." --critical ".$serviceargs["cachepages_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Cache Pages".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "checkpoints":

                        $pluginopts .= "--mode checkpoints --warning ".$serviceargs["checkpoints_warning"]." --critical ".$serviceargs["checkpoints_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Checkpoint Pages / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "connectiontime":

                        $pluginopts .= "--mode time2connect --warning ".$serviceargs["connectiontime_warning"]." --critical ".$serviceargs["connectiontime_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Connection Time".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "connectionreset":

                        $pluginopts .= "--mode connectionreset --warning ".$serviceargs["connectionreset_warning"]." --critical ".$serviceargs["connectionreset_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Connection reset / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "connections":

                        $pluginopts .= "--mode connections --warning ".$serviceargs["connections_warning"]." --critical ".$serviceargs["connections_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Connections Open".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "cpu":

                        $pluginopts .= "--mode cpu --warning ".$serviceargs["cpu_warning"]." --critical ".$serviceargs["cpu_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL CPU Utilization".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "databasepages":

                        $pluginopts .= "--mode databasepages --warning ".$serviceargs["databasepages_warning"]." --critical ".$serviceargs["databasepages_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Database Pages".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "deadlocks":

                        $pluginopts .= "--mode deadlocks --warning ".$serviceargs["deadlocks_warning"]." --critical ".$serviceargs["deadlocks_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Deadlocks / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "failedautoparams":

                        $pluginopts .= "--mode failedautoparams --warning ".$serviceargs["failedautoparams_warning"]." --critical ".$serviceargs["failedautoparams_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Failed Auto-Params / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "forwardedrecords":

                        $pluginopts .= "--mode forwardedrecords --warning ".$serviceargs["forwardedrecords_warning"]." --critical ".$serviceargs["forwardedrecords_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Forwarded Records / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "freelistalls":

                        $pluginopts .= "--mode freelistalls --warning ".$serviceargs["freelistalls_warning"]." --critical ".$serviceargs["freelistalls_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Free List Stalls / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "freepages":

                        $pluginopts .= "--mode freepages --warning ".$serviceargs["freepages_warning"]." --critical ".$serviceargs["freepages_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Free Pages".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "fullscans":

                        $pluginopts .= "--mode fullscans --warning ".$serviceargs["fullscans_warning"]." --critical ".$serviceargs["fullscans_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Full Scans / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "grantedwsmem":

                        $pluginopts .= "--mode grantedwsmem --warning ".$serviceargs["grantedwsmem_warning"]." --critical ".$serviceargs["grantedwsmem_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Granted Workspace Memory".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "indexsearches":

                        $pluginopts .= "--mode indexsearches --warning ".$serviceargs["indexsearches_warning"]." --critical ".$serviceargs["indexsearches_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Index Searches / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "latchwaits":

                        $pluginopts .= "--mode latchwaits --warning ".$serviceargs["latchwaits_warning"]." --critical ".$serviceargs["latchwaits_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Latch Waits / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "lazywrites":

                        $pluginopts .= "--mode lazywrites --warning ".$serviceargs["lazywrites_warning"]." --critical ".$serviceargs["lazywrites_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Lazy Writes / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "lockrequests":

                        $pluginopts .= "--mode lockrequests --warning ".$serviceargs["lockrequests_warning"]." --critical ".$serviceargs["lockrequests_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Lock Requests / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "locktimeouts":

                        $pluginopts .= "--mode locktimeouts --warning ".$serviceargs["locktimeouts_warning"]." --critical ".$serviceargs["locktimeouts_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Lock Timeouts / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "lockwait":

                        $pluginopts .= "--mode lockwait --warning ".$serviceargs["lockwait_warning"]." --critical ".$serviceargs["lockwait_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Lock Wait Times".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "lockwaits":

                        $pluginopts .= "--mode lockwaits --warning ".$serviceargs["lockwaits_warning"]." --critical ".$serviceargs["lockwaits_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Lock Waits / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "memory":

                        $pluginopts .= "--mode memory --warning ".$serviceargs["memory_warning"]." --critical ".$serviceargs["memory_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Memory Usage".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logins":

                        $pluginopts .= "--mode logins --warning ".$serviceargs["logins_warning"]." --critical ".$serviceargs["logins_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Logins / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "logouts":

                        $pluginopts .= "--mode logouts --warning ".$serviceargs["logouts_warning"]." --critical ".$serviceargs["logouts_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Logouts / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "longesttrans":

                        $pluginopts .= "--mode longesttrans --warning ".$serviceargs["longesttrans_warning"]." --critical ".$serviceargs["longesttrans_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Longest Transaction Running Time".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "maxwsmem":

                        $pluginopts .= "--mode maxwsmem --warning ".$serviceargs["maxwsmem_warning"]." --critical ".$serviceargs["maxwsmem_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Maximum Workspace Memory".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "memgrantsoutstand":

                        $pluginopts .= "--mode memgrantsoutstand --warning ".$serviceargs["memgrantsoutstand_warning"]." --critical ".$serviceargs["memgrantsoutstand_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Memory Grants Outstanding".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "memgrantspend":

                        $pluginopts .= "--mode memgrantspend --warning ".$serviceargs["memgrantspend_warning"]." --critical ".$serviceargs["memgrantspend_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Memory Grants Pending".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "numsuperlatches":

                        $pluginopts .= "--mode numsuperlatches --warning ".$serviceargs["numsuperlatches_warning"]." --critical ".$serviceargs["numsuperlatches_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Number of SuperLatches".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "pagelife":

                        $pluginopts .= "--mode pagelife --warning ".$serviceargs["pagelife_warning"]." --critical ".$serviceargs["pagelife_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Page Life Expectancy".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "pagelooks":

                        $pluginopts .= "--mode pagelooks --warning ".$serviceargs["pagelooks_warning"]." --critical ".$serviceargs["pagelooks_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Page Looks / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "pagereads":

                        $pluginopts .= "--mode pagereads --warning ".$serviceargs["pagereads_warning"]." --critical ".$serviceargs["pagereads_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Page Reads / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "pagesplits":

                        $pluginopts .= "--mode pagesplits --warning ".$serviceargs["pagesplits_warning"]." --critical ".$serviceargs["pagesplits_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Page Splits / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "pagewrites":

                        $pluginopts .= "--mode pagewrites --warning ".$serviceargs["pagewrites_warning"]." --critical ".$serviceargs["pagewrites_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Page Writes / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "processesblocked":

                        $pluginopts .= "--mode processesblocked --warning ".$serviceargs["processesblocked_warning"]." --critical ".$serviceargs["processesblocked_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Processes blocked".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "readahead":

                        $pluginopts .= "--mode readahead --warning ".$serviceargs["readahead_warning"]." --critical ".$serviceargs["readahead_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Readaheads / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "safeautoparams":

                        $pluginopts .= "--mode safeautoparams --warning ".$serviceargs["safeautoparams_warning"]." --critical ".$serviceargs["safeautoparams_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Safe Auto-Params / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "sqlattentionrate":

                        $pluginopts .= "--mode sqlattentionrate --warning ".$serviceargs["sqlattentionrate_warning"]." --critical ".$serviceargs["sqlattentionrate_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Sql Attention rate".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "sqlcompilations":

                        $pluginopts .= "--mode sqlcompilations --warning ".$serviceargs["sqlcompilations_warning"]." --critical ".$serviceargs["sqlcompilations_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL SQL Compilations / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "sqlrecompilations":

                        $pluginopts .= "--mode sqlrecompilations --warning ".$serviceargs["sqlrecompilations_warning"]." --critical ".$serviceargs["sqlrecompilations_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Sql Re-Compilations / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "stolenpages":

                        $pluginopts .= "--mode stolenpages --warning ".$serviceargs["stolenpages_warning"]." --critical ".$serviceargs["stolenpages_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Stolen Pages".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "superlatchdemotes":

                        $pluginopts .= "--mode superlatchdemotes --warning ".$serviceargs["superlatchdemotes_warning"]." --critical ".$serviceargs["superlatchdemotes_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL SuperLatch Demotions / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "superlatchpromotes":

                        $pluginopts .= "--mode superlatchpromotes --warning ".$serviceargs["superlatchpromotes_warning"]." --critical ".$serviceargs["superlatchpromotes_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL SuperLatch Promotions / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "tablelockescalate":

                        $pluginopts .= "--mode tablelockescalate --warning ".$serviceargs["tablelockescalate_warning"]." --critical ".$serviceargs["tablelockescalate_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Table Lock Escalations / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                             "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "targetpages":

                        $pluginopts .= "--mode targetpages --warning ".$serviceargs["targetpages_warning"]." --critical ".$serviceargs["targetpages_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Target Pages".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "targetsrvmem":

                        $pluginopts .= "--mode targetsrvmem --warning ".$serviceargs["targetsrvmem_warning"]." --critical ".$serviceargs["targetsrvmem_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Target Server Memory".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "totallatchwait":

                        $pluginopts .= "--mode totallatchwait --warning ".$serviceargs["totallatchwait_warning"]." --critical ".$serviceargs["totallatchwait_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Total Latch Wait Time".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "totalpages":

                        $pluginopts .= "--mode totalpages --warning ".$serviceargs["totalpages_warning"]." --critical ".$serviceargs["totalpages_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Total Pages".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "totalsrvmem":

                        $pluginopts .= "--mode totalsrvmem --warning ".$serviceargs["totalsrvmem_warning"]." --critical ".$serviceargs["totalsrvmem_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Total Server Memory".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "unsafeautoparams":

                        $pluginopts .= "--mode unsafeautoparams --warning ".$serviceargs["unsafeautoparams_warning"]." --critical ".$serviceargs["unsafeautoparams_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Unsafe Auto-Params / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "usercons":

                        $pluginopts .= "--mode usercons --warning ".$serviceargs["usercons_warning"]." --critical ".$serviceargs["usercons_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL User Connections".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "workfilescreated":

                        $pluginopts .= "--mode workfilescreated --warning ".$serviceargs["workfilescreated_warning"]." --critical ".$serviceargs["workfilescreated_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Workfiles Created / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "worktablescacheratio":

                        $pluginopts .= "--mode worktablescacheratio --warning ".$serviceargs["worktablescacheratio_warning"]." --critical ".$serviceargs["worktablescacheratio_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Worktables From Cache Ratio".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "worktablescreated":

                        $pluginopts .= "--mode worktablescreated --warning ".$serviceargs["worktablescreated_warning"]." --critical ".$serviceargs["worktablescreated_critical"]."";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "MSSQL Worktables Created / Sec".$instancetext,
                            "use" => "xiwizard_mssqlserver_service",
                            "check_command" => "check_xi_mssql_server2!".$pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "process":
                        foreach ($args as $i => $servicearg) {
                            if (!array_key_exists('process', $service) || empty($servicearg['counter_name'])) {
//echo("SKIP 1<br>");
                                continue;
                            } else if ($service['process'][$i] == 'off') {
//echo("SKIP 2<br>");
                                continue;
                            }
//echo("ADDING<br>");

                            $pluginCustomOpts  = '--mode \'custom\' --warning '.$servicearg["warning"].' --critical '.$servicearg["critical"].' ';
                            $pluginCustomOpts .= '--custom \'("counter_name" : "'.$servicearg["counter_name"].'", "instance_name" : "'.$servicearg["instance_name"].'", "unit" : "'.$servicearg["unit"].'", "modifier" : "'.$servicearg["modifier"].'", "ring_buffer_type" : "'.$servicearg["ring_buffer_type"].'", "xpath" : "'.$servicearg["xpath"].'")\'';

                            $serviceDescription = $servicearg['display_name'];

                            if (empty($servicearg['display_name'])) {
                                # e.g. Cache Hit Ratio to 'cachehitratio'
                                $serviceDescription = ucwords(str_replace("%", "Pct", $servicearg["counter_name"]));
                                $serviceDescription = (!empty($servicearg["instance_name"]) ? $servicearg["instance_name"]." " : "")."MSSQL ".$serviceDescription;
                            }

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $serviceDescription,
                                "use" => "xiwizard_mssqlserver_service",
                                "_xiwizard" => $wizard_name,
                                "check_command" => "check_xi_mssql_server2!".$pluginopts.$pluginCustomOpts,
                            );
                        }

                        break;

                    default:

                        break;
                }
            }

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:

            break;
    }

    return $output;
}


