<?php
//
// SSH Proxy Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

sshproxy_configwizard_init();

function sshproxy_configwizard_init()
{
    $name = "sshproxy";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.1.0",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor a remote Linux, Unix, or Mac OS/X machine using SSH."),
        CONFIGWIZARD_DISPLAYTITLE => _("SSH Proxy"),
        CONFIGWIZARD_FUNCTION => "sshproxy_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "ssh.png",
        CONFIGWIZARD_FILTER_GROUPS => array('linux','otheros'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function sshproxy_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "sshproxy";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");
            $osdistro = grab_array_var($inargs, "osdistro", "");
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address = nagiosccm_replace_user_macros($address);
            $osdistro = grab_array_var($inargs, "osdistro", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $osdistro = grab_array_var($inargs, "osdistro", "");
            $ha = @gethostbyaddr($address);
            $selectedhostconfig = grab_array_var($inargs, "selectedhostconfig", "");
            $operation = grab_array_var($inargs, "operation", "new");

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);
            $password = "";
            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            if (!is_array($services)) {
                $services_default = array(
                    "ping" => 1,
                    "commands" => array(),
                );

                $services_default["commands"][0] = "on";
                $services_default["commands"][1] = "on";
                $services_default["commands"][2] = "on";
                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs = "";
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            if (!is_array($serviceargs)) {
                $serviceargs_default = array(
                    "commands" => array(),
                );

                for ($x = 0; $x < 6; $x++) {
                    if ($x == 0) {
                        $serviceargs_default['commands'][$x]['command'] = '/usr/local/nagios/libexec/check_disk /';
                        $serviceargs_default['commands'][$x]['args'] = '';
                        $serviceargs_default['commands'][$x]['name'] = 'Root Disk Space';
                    } else if ($x == 1) {
                        $serviceargs_default['commands'][$x]['command'] = '/usr/local/nagios/libexec/check_users -w 5 -c 10';
                        $serviceargs_default['commands'][$x]['args'] = '';
                        $serviceargs_default['commands'][$x]['name'] = 'Current Users';
                    } else if ($x == 2) {
                        $serviceargs_default['commands'][$x]['command'] = '/usr/local/nagios/libexec/check_procs -w 150 -c 170';
                        $serviceargs_default['commands'][$x]['args'] = '';
                        $serviceargs_default['commands'][$x]['name'] = 'Total Processes';
                    } else if ($x == 3) {
                        $serviceargs_default['commands'][$x]['command'] = '/usr/local/nagios/libexec/check_load -w 2,1.5,1 -c 4,3,2';
                        $serviceargs_default['commands'][$x]['args'] = '';
                        $serviceargs_default['commands'][$x]['name'] = 'Total CPU Load';
                    } else if ($x == 4) {
                        $serviceargs_default['commands'][$x]['command'] = '/usr/local/nagios/libexec/check_swap -w 30% -c 10%';
                        $serviceargs_default['commands'][$x]['args'] = '';
                        $serviceargs_default['commands'][$x]['name'] = 'Free Swap Space';
                    }
                    // } else {
                    //     $serviceargs_default['commands'][$x]['command'] = '';
                    //     $serviceargs_default['commands'][$x]['args'] = '';
                    //     $serviceargs_default['commands'][$x]['name'] = '';
                    //     $services['commands'][$x] = ""; // defaults for checkboxes, enter on to be checked by default
                    // }
                }

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $osdistro = grab_array_var($inargs, "osdistro", "");

            // check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = _("Invalid host name.");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $osdistro = grab_array_var($inargs, "osdistro", "");

            $services = grab_array_var($inargs, "services");
            if ($services == "") {
                $services_serial = grab_array_var($inargs, "services_serial");
                if ($services_serial != "") {
                    $services = json_decode(base64_decode($services_serial), true);
                }
            }

            $serviceargs = grab_array_var($inargs, "serviceargs");
            if ($serviceargs == "") {
                $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");
                if ($serviceargs_serial != "") {
                    $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
                }
            }

            $output = '

        <input type="hidden" name="ip_address" value="' . $address . '">
        <input type="hidden" name="hostname" value="' . $hostname . '">
        <input type="hidden" name="osdistro" value="' . $osdistro . '">
        <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
        <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $osdistro = grab_array_var($inargs, "osdistro", "");
            $hostaddress = $address;

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["osdistro"] = $osdistro;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            $icon = sshproxy_configwizard_get_distro_icon($osdistro);

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_linuxserver_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => $icon,
                    "statusmap_image" => $icon,
                    "_xiwizard" => $wizard_name,
                );
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($svc) {

                    case "ping":
                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Ping",
                            "use" => "xiwizard_linuxserver_ping_service",
                            "_xiwizard" => $wizard_name,
                        );
                        break;


                    case "commands":

                        $enabledcmds = $svcstate;
                        foreach ($enabledcmds as $pid => $pstate) {

                            $pname = $serviceargs["commands"][$pid]["command"];
                            $pdesc = $serviceargs["commands"][$pid]["name"];

                            $checkcommand = "check_xi_by_ssh!-C \"" . $pname . "\"";

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $pdesc,
                                "use" => "generic-service",
                                "check_command" => $checkcommand,
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;

                    default:
                        break;
                }
            }

            // echo "OBJECTS:<BR>";
            // print_r($objs);
            // exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


/**
 * @param $osdistro
 *
 * @return string
 */
function sshproxy_configwizard_get_distro_icon($osdistro)
{

    $icon = "linux.png";

    switch ($osdistro) {
        case "Solaris":
            $icon = "solaris.png";
            break;
        case "AIX":
            $icon = "aix.png";
            break;
        case "HP-UX":
            $icon = "hp-ux.png";
            break;

        case "FreeBSD":
            $icon = "freebsd2.png";
            break;
        case "NetBSD":
            $icon = "netbsd.png";
            break;
        case "OpenBSD":
            $icon = "openbsd.png";
            break;
        case "Windows":
            $icon = "windowsxp.png";
            break;
        case "RHEL":
            $icon = "redhat.png";
            break;
        case "Fedora":
            $icon = "fedora.png";
            break;
        case "CentOS":
            $icon = "centos.png";
            break;
        case "Ubuntu":
            $icon = "ubuntu.png";
            break;
        case "Debian":
            $icon = "debian.png";
            break;
        case "OpenSUSE":
            $icon = "opensuse.png";
            break;
        case "SUSE":
            $icon = "suse_enterprise.png";
            break;
        default:
            break;
    }

    return $icon;
}
