<?php
//
// VSphere Config Wizard
// Copyright (c) 2008-2025 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

vsphere_configwizard_init();

// Originally this was just VSPHERE_SERVICENAMES. It was split up into two nested arrays as the subcommands for the commands can differ based on if it is viewing guests or hosts on VMWare.
define('VSPHERE_SERVICENAMES_HOST',
    json_encode(array(
            array('CPU', TRUE, _('CPU Usage'), '%, MHz - Example: 30,200', 'Usage', 'UsageMHZ'),
            array('MEM', TRUE, _('Memory'), '%, MB, Overhead, Swapped, Memctl, Consumed - Example: 30,200,10,10,5,5000', 'Usage', 'UsageMB', 'Swap', 'OverHead', 'MemCtl'),
            array('NET', TRUE, _('Networking'), 'Recive, Send, nics connected - Example: 5,5,3', 'Usage', 'Recieve', 'Send', 'NIC'),
            array('IO', TRUE, _('Input / Output'), 'Usage, Read, Write - Example: 1,0.5,0.1', 'Usage', 'Read', 'Write'),
            array('VMFS', FALSE, _('Datastore Free'), 'Space free for each datastore - Example for 2 datastores: 400000,250000', 'Name'),
            array('RUNTIME', TRUE, _('VM Status'), 'State, Connection , Uptime - Thresholds not currently implemented for this metric', 'Con', 'Health', 'StorageHealth', 'Temperature', 'Sensor', 'Maintenance', 'ListVm', 'Status', 'Issues'),
            array('SERVICE', FALSE, _('Services'), 'States of the services - Thresholds not currently implemented for this metric', 'Names'),
            // array('RUNTIME', TRUE, 'VM Status', 'State, Connection , Uptime - Example: Down,Disconnected,50d', 'Con', 'Health', 'StorageHealth', 'Temperature', 'Sensor', 'Maintenance', 'ListVm', 'Status', 'Issues'),
            // array('SERVICE', FALSE, 'Services', 'States of the services - Example: Down,Down,Down,Down,Down,Up,Up,Down,...etc', 'Names'),
        )
    )
);

define('VSPHERE_SERVICENAMES_GUESTS',
    json_encode(array(
            array('CPU', TRUE, _('CPU Usage'), '%, MHz, wait, ready - Example: 30,350,50000,50', 'Usage', 'UsageMHZ', 'Wait', 'Ready'),
            array('MEM', TRUE, _('Memory'), '%, MB, Overhead, Swapped, Swap in, Swap out, Memctl, Consumed - Example: 40,300,50,5,5,5,10,5000', 'Usage', 'UsageMB', 'Swap', 'OverHead', 'Active', 'MemCtl'),
            array('NET', TRUE, _('Networking'), 'Recieve, Send - Example: 5,5', 'Usage', 'Recieve', 'Send'),
            array('IO', TRUE, _('Input / Output'), 'Usage, Read, Write - Example: 5,5,5', 'Usage', 'Read', 'Write'),
            array('VMFS', FALSE, _('Datastore Free'), 'Space free for each datastore - Example for 2 datastores: 400000,250000', 'Name'),
            array('RUNTIME', TRUE, _('VM Status'), 'State, Status, Connection, Console, Max cpu, Max memory, Tools - Thresholds not currently implemented for this metric', 'Con', 'Cpu', 'Mem', 'State', 'Status', 'ConsoleConnetions', 'Guest', 'Tools', 'Issues'),
            // array('RUNTIME', TRUE, 'VM Status', 'State, Status, Connection, Console, Max cpu, Max memory, Tools, Config issues - Example: DOWN,red,Disconnected,1,3300:,3500:,down,1', 'Con', 'Cpu', 'Mem', 'State', 'Status', 'ConsoleConnetions', 'Guest', 'Tools', 'Issues'),
        )
    )
);

define('VSPHERE_BASICDATA',
    json_encode(array(
            array(
                array("wng", "warning", "Warning"),
                array("crt", "critical", "Critical"),
            ),
            array(
                array("low", "low", "%s Below:"),
                array("hi", "high", "Above:"),
            )
        )
    )
);

define('VSPHERE_HOSTINPUTF', '');

function vsphere_configwizard_init()
{
    $name = 'vsphere';
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "1.1.0",
        CONFIGWIZARD_IS_PREMIUM => TRUE,
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _('Monitor a VMware host or guest VM.'),
        CONFIGWIZARD_DISPLAYTITLE => _('vSphere'),
        CONFIGWIZARD_FUNCTION => 'vsphere_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => 'vsphere.png',
        CONFIGWIZARD_REQUIRES_VERSION => 70000
    );
    register_configwizard($name, $args);
}

/**
 * Get the Linux distribution and package manager
 * 
 */
function getPackageManager() {

    $ini = parse_ini_file('/usr/local/nagiosxi/var/xi-sys.cfg');
    $dist = $ini['dist'];

    if ($dist == "el9" || $dist == "el8") {
        $packageManager = "dnf";
    }
    elseif ($dist == "debian12" || $dist == "debian11" || $dist == "ubuntu20" || $dist == "ubuntu22" || $dist == "ubuntu24") {
        $packageManager = "apt";
    }
    else {
        $packageManager = "yum";
    }

    return $packageManager;
}

/**
 * Checks to verify that VMware SDK is installed
 * 
 */
function vsphere_configwizard_check_prereqs()
{
    $checkCmd = "python3 --version";
    $checkOutput = shell_exec($checkCmd);
    if (strpos($checkOutput, 'Python 3') === false){
        return false;
    }

    $checkCmd = "python3 -m pip show pyvmomi";
    $checkOutput = shell_exec($checkCmd);

    if (strpos($checkOutput, 'Name: pyvmomi') === false){
        //// Trying to install here is really slow and delays the wizard loading by a long time. TODO: better solution?
        // $installCmd = "python3 -m pip install --upgrade pyvmomi";
        // $installOutput = shell_exec($installCmd);
        // if (strpos($installOutput, 'Successfully installed') === false) {
        //     return false;
        // }
        return false;
    }
    return true;
}

/**
 * @param null $inargs
 *
 * @return array
 */
function vsphere_configwizard_parseinargs($inargs = null)
{
    // Get variables that were passed to us
    $address = grab_array_var($inargs, 'ip_address', '');
    $hostname = grab_array_var($inargs, 'hostname', '');
    $ha = '';
    if ($hostname == '') {
        $ha = $address == '' ? '' : @gethostbyaddr($address);
        if ($ha == '')
            $ha = $address;
    }
    $hostname = grab_array_var($inargs, 'hostname', $ha);
    $type = grab_array_var($inargs, 'type', 'host');
    $username = grab_array_var($inargs, 'username', '');
    $password = grab_array_var($inargs, 'password', '');

    $services_serial = grab_array_var($inargs, 'services_serial', '');
    $serviceargs_serial = grab_array_var($inargs, 'serviceargs_serial', '');
    $guests_serial = grab_array_var($inargs, 'guests_serial', '');

    $services = json_decode(base64_decode($services_serial), true);
    $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
    $guests = json_decode(base64_decode($guests_serial), true);
    if (!is_array($services))
        $services = array();
    if (!is_array($serviceargs))
        $serviceargs = array();
    if (!is_array($guests))
        $guests = array();
    $srvlock = 0;
    $guestlock = 0;

    foreach (array_keys($inargs) as $argu) {
        if ( preg_match('/^activate_(.*)$/', $argu, $matches)) {
            if (!$guestlock) {
                $guests = array();
                $guestlock = -1;
            }
            $argt = base64_decode($matches[1]);
            $guests[$argt] = grab_array_var($inargs, "alias_".$matches[1], $argt);
        }
        if (preg_match('/^service_(.*)$/', $argu, $matches)) {
            if (!$srvlock) {
                $services = array();
                $srvlock = -1;
            }
            $services[$matches[1]] = TRUE;
        }
        if (preg_match('/^serviceargs_([^-]*)-(.*)$/', $argu, $matches)) {
            $argt = $matches[1] . '_' . $matches[2];
            if (array_search($argt, $serviceargs) === FALSE)
                $serviceargs[$argt] = grab_array_var($inargs, $argu, '');
        }
    }
    if (empty($serviceargs)) {
        $serviceargs = grab_array_var($inargs, 'serviceargs', array());
    }

    unset ($argu);

    return array($hostname, $address, $type, $username, $password, $services, $serviceargs, $guests);
}

/**
 * @param $output
 * @param $s
 * @param $services
 * @param $serviceargs
 * @param $mode
 */
function vsphere_configwizard_pushcheckboxandargs($s, $services, $serviceargs, $mode, $thresholdsdisabled = false)
{
    $sl = strtolower($s[0]);

    $tdstyles = 'vertical-align:middle;';
    $checkboxandargs = '<tr style="vertical-align: middle;">
                    <td style="' . $tdstyles . '">
                        <input type="checkbox" 
                            class="service_checkbox" 
                            id="ckhbx_' . encode_form_val($sl) . '" 
                            name="service_' . encode_form_val($sl) . '" ' . 
                            (in_array($sl, $services) ? 'checked="yes"' : '') . ' 
                            style="margin-top: 8px;">
                        </input>
                    </td>
                    <td style="' . $tdstyles . '">
                        <label for="serviceargs['. $sl .'_warning]" class="form-label">'. encode_form_val($s[2]) .'</label>
                        <i class="fa fa-question-circle" title="'. $s[3] .'" style="margin-left: 5px;"></i>
                    </td>';

                    if (!$thresholdsdisabled) {
                        $checkboxandargs .= '' .
                        '<td style="' . $tdstyles . '">
                            <div class="input-group">
                                <span class="input-group-text">
                                    <i title="'._('Warning Threshold') .'" class="material-symbols-outlined md-warning md-18 md-400 md-middle">warning</i>
                                </span>
                                <input type="text" 
                                    id="serviceargs['. $sl .'_warning]" 
                                    name="serviceargs['. $sl .'_warning]" 
                                    value="'. encode_form_val($serviceargs[$sl . '_warning']) .'" 
                                    class="form-control">
                            </div>
                        </td>
                        <td style="' . $tdstyles . '">
                            <div class="input-group">
                                <span class="input-group-text">
                                    <i title="'._('Critical Threshold') .'" class="material-symbols-outlined md-critical md-18 md-400 md-middle">error</i>
                                </span>
                                <input type="text" 
                                    id="serviceargs['. $sl .'_critical]" 
                                    name="serviceargs['. $sl .'_critical]" 
                                    value="'. encode_form_val($serviceargs[$sl . '_critical']) .'" 
                                    class="form-control">
                            </div>
                        </td>';
                    }
                $checkboxandargs .= '</tr>';
        echo $checkboxandargs;



    // TODO: If we add subcommands to the wizard this code will need to be added above to add a metrics select box. Currently it doesn't make sense to leave metrics in without subcommands as the different metrics should have different metric values.
    //     <div class="col-sm-3 mt-0">
    //     <label>'. _('Metrics').'</label>
    //     <select name="serviceargs['.$sl.'_metrics]" id="serviceargs['. $sl . '_metrics]" class="form-control">
    //         <option value="percentage">%</option>
    //         <option value="B">B</option>
    //         <option value="KB">KB</option>
    //         <option value="Kib">Kib</option>
    //         <option value="MB">MB</option>
    //         <option value="Mib">Mib</option>
    //         <option value="GB">GB</option>
    //         <option value="Gib">Gib</option>
    //         <option value="TB">TB</option>
    //         <option value="Tib">Tib</option>
    //     </select>
    // </div>
}

/**
 * @param $serviceargs
 * @param $svcl
 *
 * @return string
 */
function vsphere_configwizard_getrangeargs($serviceargs, $svcl)
{
    $ret = [];

    $svclargs = array();
    foreach ($serviceargs as $key => $value) {
        if (strpos($key, $svcl) === 0) {
            $svclargs[$key] = $value;
        }
    }
    $warning = $svclargs[$svcl . '_warning'];
    $critical = $svclargs[$svcl . '_critical'];
    if (!($warning == ''))
        $ret[] = '-w' . $warning;
    if (!($critical == ''))
        $ret[] = '-c' . $critical;

    // TODO: if subcommands are added we will want to add metrics back in. We will need to change how critical/warning thresholds are done as they will no longer be given in a csv with subcommands. May look like the commented code below.
    // $metrics = $svclargs[$svcl . '_metrics'];
    // if (!($warning == ''))
    //     $ret .= ' -w ' . $warning;
    // if (!($critical == ''))
    //     $ret .= ' -c ' . $critical;
    // used for subcommand metrics
    // if (!($metrics == 'percentage'))
    //     $ret .= ' -U ' . $metrics;
    return $ret;
}


/**
 * @param $objs
 * @param $type
 * @param $hostname
 * @param $address
 * @return array
 */
function vsphere_configwizard_makehost(&$objs, $type, $hostname, $address)
{
    return array(
        'type' => OBJECTTYPE_HOST,
        'use' => 'xiwizard_generic_host',
        'host_name' => $hostname,
        'address' => $address,
        'icon_image' => 'vsphere.png',
        'statusmap_image' => 'vsphere.png',
        '_xiwizard' => 'vsphere',
    );
}

/**
 * @param $objs
 * @param $hostname
 * @param $address
 * @param $type
 * @param $services
 * @param $serviceargs
 * @param $guests
 */
function vsphere_configwizard_makeservices(&$objs, $hostname, $address, $type, $services, $serviceargs, $guests)
{
    // TODO: If we are doing subcommands we will need to change how we loop through the services. The current version only checks
    // the main command itself and doesnt go into the sub commands, i.e. -l cpu vrs -l cpu -s Usage. -s being the subcommand for cpu
    $fil = get_root_dir() . '/etc/components/vsphere/' . preg_replace("/[ '.\:_-]/", '_', $hostname) . '_auth.txt';
    if (!host_exists($hostname))
        $objs[] = vsphere_configwizard_makehost($objs, $type, $hostname, $address);
    switch ($type) {
        case 'guest':
            foreach ($guests as $guestaddress => $guestname) {
                $guestname = str_replace(',', '', $guestname);

                // see which services we should monitor
                foreach (json_decode(VSPHERE_SERVICENAMES_GUESTS, true) as $s) {
                    $sl = strtolower($s[0]);
                    if (array_key_exists($sl, $services)) {
                        $warn = vsphere_configwizard_getrangeargs($serviceargs, $sl);
                        $objs[] = array(
                            'type' => OBJECTTYPE_SERVICE,
                            'host_name' => $hostname,
                            'service_description' => "{$guestname} {$s[2]}",
                            'use' => 'xiwizard_generic_service',
                            'check_command' => "check_vsphere_guest_xi!$fil!$guestaddress!" . $s[0] . "!$warn[0]" . "!$warn[1]",
                            '_xiwizard' => 'vsphere',
                        );
                    }
                }
            }
            break;

        case 'host':

            // see which services we should monitor
            foreach (json_decode(VSPHERE_SERVICENAMES_HOST, true) as $s) {
                $sl = strtolower($s[0]);
                if (array_key_exists($sl, $services)) {
                    $warn = vsphere_configwizard_getrangeargs($serviceargs, $sl);
                    $objs[] = array(
                        'type' => OBJECTTYPE_SERVICE,
                        'host_name' => $hostname,
                        'service_description' => $s[2] . " for vSphere Host",
                        'use' => 'xiwizard_generic_service',
                        'check_command' => 'check_vsphere_host_xi!' . $fil . '!' . $s[0] . "!$warn[0]" . "!$warn[1]",
                        '_xiwizard' => 'vsphere',
                    );
                }
            }

            // Commands below are for datastores to be checked.
            foreach ($guests as $guest) {
                $objs[] = array(
                    'type' => OBJECTTYPE_SERVICE,
                    'host_name' => $hostname,
                    'service_description' => "Datastore $guest for vSphere Host",
                    'use' => 'xiwizard_generic_service',
                    'check_command' => 'check_vsphere_datastore_xi!' . $fil . '!' . $guest . '!',
                    '_xiwizard' => 'vsphere',
                );
            }
            break;

        default:
            break;
    }
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function vsphere_configwizard_func($mode = '', $inargs = null, &$outargs = null, &$result = null)
{
    // Initialize return code and output
    $result = 0;
    $output = '';

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            
        $packageManager = getPackageManager();

            $output = '';

            if (vsphere_configwizard_check_prereqs() == false) {
                $output .= '<div class="message" style="margin-top: 20px;">
                                <ul class="actionMessage">
                                    <li class="mb-1">
                                    '._('It appears as though you have not installed the necessary Python packages. Once you have Python3 installed, you will need to install the pyvmomi package.<br>
                                    You can install the pyvmomi package by running the following commands:').'</li>
                                    <pre>
'                                       . 'python3 -m pip install --upgrade pip
'                                       . 'python3 -m pip install --upgrade setuptools wheel
'                                       . 'python3 -m pip install --upgrade importlib-metadata
'                                       . ''.$packageManager.' install python3-pyvmomi -y</pre>
                                    <li>'._('Please refer to the following documentation for more information:').' <strong><a href="https://assets.nagios.com/downloads/nagiosxi/docs/Monitoring-VSphere.pdf" target="_blank">'._('Monitoring vSphere with Nagios XI').'</a></strong> '._('documentation').'</a></strong> </li>
                                </ul>
                            </div>';
            
            } else {
                list($hostname, $address, $type, $username, $password, $services, $serviceargs, $guests) = vsphere_configwizard_parseinargs($inargs);

                $output .= '
<input type="hidden" name="services_serial" value="' . encode_form_val(base64_encode(json_encode($services))) . '">
<input type="hidden" name="serviceargs_serial" value="' . encode_form_val(base64_encode(json_encode($serviceargs))) . '">
<input type="hidden" name="guests_serial" value="' . encode_form_val(base64_encode(json_encode($guests))) . '">

<h5 class="ul">' . _('VMware Information') . '</h5>
<table class="table table-condensed table-no-border table-auto-width">
    <tr>
        <td class="vt">
            <label>' . _('Address') . ':</label>
        </td>
        <td>
            <input type="text" size="30" name="ip_address" id="ip_address" value="' . encode_form_val($address) . '" class="form-control" autocomplete="ip_address">
            <div class="subtext">' . _('The IP address or FQDNS name of the VCenter (server) or ESXi host you would like to monitor.') . '</div>
        </td>
    </tr>
    <tr>
        <td class="vt">
            <label>' . _('Username') . ':</label>
        </td>
        <td>
            <input type="text" size="30" name="username" id="username" value="' . encode_form_val($username) . '" class="form-control">
            <div class="subtext">' . _('The username used to authenticate on the VMware server.') . '</div>
        </td>
    </tr>
    <tr>
        <td class="vt">
            <label>' . _('Password') . ':</label>
        </td>
        <td>
            <input type="password" size="30" name="password" id="password" value="' . encode_form_val($password) . '" class="form-control">
            <div class="subtext">' . _('The password used to authenticate on the VMware server.') . '</div>
        </td>
    </tr>
    <tr>
        <td>
            <label>' . _('Monitoring Mode') . ':</label>
        </td>
        <td>    
            ' . _('Would you like to monitor the VMware host (server) or a guest VM?') . '
            <div class="pad-t5">
                <div style="align-items: center;">
                    <label><input style="margin-left:0px !important;margin-right: 2px;" type="radio" name="type" value="host" ' . ($type === "host" ? ' checked="yes"' : '') . '>' . _('Monitor the VMware host') . '</label>
                </div>
                <div style="align-items: center;">
                    <label><input style="margin-left:0px !important;margin-right: 2px;" type="radio" name="type" value="guest" ' . ($type === "guest" ? ' checked="yes"' : '') . '>' . _('Monitor a guest VM on the VMWare host') . '</label>
                </div>
            </div>
        </td>
    </tr>
</table>';

            }
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            list($hostname, $address, $type, $username, $password, $services, $serviceargs, $guests) = vsphere_configwizard_parseinargs($inargs);

            // check for form errors
            $errors = 0;
            $errmsg = array();
            
            if (vsphere_configwizard_check_prereqs() == false) {
                $errmsg[$errors++] = _('Required software components are missing.');
            } else {
                if (have_value($address) == false) {
                    $errmsg[$errors++] = _('No address specified.');
                }
                if (have_value($username) == false) {
                    $errmsg[$errors++] = _('Username not specified.');
                }
                if (have_value($password) == false) {
                    $errmsg[$errors++] = _('Password not specified.');
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            list($hostname, $address, $type, $username, $password, $services, $serviceargs, $guests) = vsphere_configwizard_parseinargs($inargs);

            $output = '
<input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
<input type="hidden" name="type" value="' . encode_form_val($type) . '">
<input type="hidden" name="username" value="' . encode_form_val($username) . '">
<input type="hidden" name="password" value="' . encode_form_val($password) . '">

<h5 class="ul">' . _('VMware Details') . '</h5>
<table class="table table-condensed table-no-border table-auto-width">
    <tr>
        <td>
            <label>' . _('VMware Mode') . ':</label>
        </td>
        <td>
            ' . ucfirst(encode_form_val($type)) . '
        </td>
    </tr>
    <tr>
        <td>
            <label>' . _('Address') . ':</label>
        </td>
        <td>
            <input type="text" size="20" name="ip_address" id="ip_address" value="' . encode_form_val($address) . '" class="form-control" disabled>
        </td>
    </tr>
    <tr>
        <td class="vt">
            <label>' . _('Host Name') . ':</label>
        </td>
        <td>
            <input type="text" size="20" name="hostname" id="hostname" value="' . encode_form_val($hostname) . '" class="form-control">
            <div class="subtext">' . _('The name you\'d like to have associated with this host.') . '</div>
        </td>
    </tr>
</table>';

            switch ($type) {
                case 'guest':
                    $services = ['cpu', 'mem', 'net', 'io', 'vmfs', 'runtime']; // List of default services to be checked for monitoring on guests
                    ob_start();
                    include __DIR__ . '/steps/step2_guest.php';
                    $output .= ob_get_clean();
                    break;
                case "host":
                    $services = ['cpu', 'mem', 'net', 'io', 'vmfs', 'runtime', 'service']; // List of default services to be checked for monitoring on hosts
                    ob_start();
                    include __DIR__ . '/steps/step2_host.php';
                    $output .= ob_get_clean();
                    break;
                default:
                    break;
            }
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            list($hostname, $address, $type, $username, $password, $services, $serviceargs, $guests) = vsphere_configwizard_parseinargs($inargs);

            // User macros
            $address = nagiosccm_replace_user_macros($address);
            $hostname = nagiosccm_replace_user_macros($hostname);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            // Check for errors
            $errors = 0;
            $errmsg = array();
            if (is_valid_host_name($hostname) === false) {
                $errmsg[$errors++] = _('Invalid host name.');
            }

            foreach ($services as $s) {
                if (is_valid_service_name($s) === false) {
                    $errmsg[$errors++] = sprintf(_('Invalid service name') . " %s", $s);
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            list($hostname, $address, $type, $username, $password, $services, $serviceargs, $guests) = vsphere_configwizard_parseinargs($inargs);

            $output = ' <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
                        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
                        <input type="hidden" name="type" value="' . encode_form_val($type) . '">
                        <input type="hidden" name="username" value="' . encode_form_val($username) . '">
                        <input type="hidden" name="password" value="' . encode_form_val($password) . '">
                        <input type="hidden" name="services_serial" value="' . encode_form_val(base64_encode(json_encode($services))) . '">
                        <input type="hidden" name="serviceargs_serial" value="' . encode_form_val(base64_encode(json_encode(
                    $serviceargs))) . '">
                        <input type="hidden" name="guests_serial" value="' . encode_form_val(base64_encode(json_encode($guests))) . '">';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            list($hostname, $address, $type, $username, $password, $services, $serviceargs, $guests) = vsphere_configwizard_parseinargs($inargs);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr['hostname'] = $hostname;
            $meta_arr['ip_address'] = $address;
            $meta_arr['type'] = $type;
            $meta_arr['services'] = $services;
            $meta_arr['serviceargs'] = $serviceargs;
            $meta_arr['guests'] = $guests;
            save_configwizard_object_meta('vsphere', $hostname, '', $meta_arr);

            $objs = array();

            // write auth data file
            $fil = get_root_dir() . '/etc/components/vsphere';
            if (!file_exists($fil))
                mkdir($fil, 0770);
            $fil .= '/' . preg_replace('/[ .\:_-]/', '_', $hostname) . '_auth.txt';

            $fh = fopen($fil, 'w+');
            if ($fh) {
                fputs($fh, 'username=' . $username . "\npassword=" . $password . '');
                fclose($fh);
            }

            vsphere_configwizard_makeservices($objs, $hostname, $address, $type, $services, $serviceargs, $guests);

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;
            // echo "<pre style='color: white;'>";
            // var_dump($objs);
            // echo "</pre>";

            break;

        default:

            break;
    }

    return $output;
}
