<?php
//
// Windows Event Log Config Wizard
// Copyright (c) 2010-2020 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

windowseventlog_configwizard_init();

function windowseventlog_configwizard_init()
{
    $name = "windowseventlog";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.2",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _('Monitor Windows event logs via NCPA.'),
        CONFIGWIZARD_DISPLAYTITLE => _('Windows Event Log'),
        CONFIGWIZARD_FUNCTION => 'windowseventlog_configwizard_func',
        CONFIGWIZARD_PREVIEWIMAGE => 'windowseventlog.png',
        CONFIGWIZARD_FILTER_GROUPS => array('windows'),
        CONFIGWIZARD_REQUIRES_VERSION => 500
    );
    register_configwizard($name, $args);
}


/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function windowseventlog_configwizard_func($mode = "", $inargs = null, &$outargs, &$result)
{
    $wizard_name = "windowseventlog";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);

            $output = '
            <h5 class="ul">' . _('Setup NCPA') . '</h5>
            <p>' . _('The agent should be installed before running this wizard.') . '</p>
            <ul style="padding: 0 0 0 30px;">
                <li><a href="https://www.nagios.org/ncpa#downloads" target="_blank">' . _('Download the latest version of NCPA') . '</a> ' . _('for the system you would like to monitor') . '</li>
                <li>' . _('Follow the') . ' <a href="https://www.nagios.org/ncpa/getting-started.php" target="_blank">' . _('installation instructions') . '</a> (<a href="https://assets.nagios.com/downloads/ncpa/docs/Installing-NCPA.pdf" target="_blank">' . _('PDF version') . '</a>) ' . _('and configure the token for the agent') . '</li>
            </ul>

            <h5 class="ul">' . _('Connect to NCPA') . '</h5>
            <table class="table table-condensed table-no-border table-auto-width table-padded">
                <tr>
                    <td class="vt"></td>
                    <td class="checkbox">
                        <label>
                            <input type="checkbox" name="no_ssl_verify" value="1" ' . is_checked($no_ssl_verify, 1) . '>
                            ' . _("Do not verify SSL certificate") . '
                        </label>
                    </td>
                </tr>
                <tr>
                    <td class="vt"><label>' . _('Address') . ':</label></td>
                    <td>
                        <input type="text" size="40" name="ip_address" value="' . encode_form_val($ip_address) . '" class="textfield usermacro-detection form-control" autocomplete="off">
                        <div class="subtext">' . _('The IP address or FQDNS name used to connect to NCPA') . '.</div>
                    </td>
                </tr>
                <tr>
                    <td class="vt"><label>' . _('Port') . ':</label></td>
                    <td>
                        <input type="text" size="5" name="port" value="' . encode_form_val($port) . '" class="textfield usermacro-detection form-control">
                        <div class="subtext">' . _('Port used to connect to NCPA. Defaults to port 5693') . '.</div>
                    </td>
                </tr>
                <tr>
                    <td class="vt"><label>' . _('Token') . ':</label></td>
                    <td>
                        <input type="text" size="20" name="token" id="token" value="' . encode_form_val($token) . '" class="textfield usermacro-detection form-control" autocomplete="off">
                        <div class="subtext">' . _('Authentication token used to connect to NCPA') . '.</div>
                    </td>
                </tr>
            </table>';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "5693");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($ip_address) == false) {
                $errmsg[$errors++] = _("No address specified.");
            }
            if (have_value($port) == false) {
                $errmsg[$errors++] = _("No port number specified.");
            }

            // Test the connection if no errors
            if (empty($errors)) {

                $ip_address_replaced = nagiosccm_replace_user_macros($ip_address);
                $port_replaced = nagiosccm_replace_user_macros($port);
                $token_replaced = nagiosccm_replace_user_macros($token);

                // The URL we will use to query the NCPA agent, and do a walk
                // of all monitorable items.
                $query_url = "https://{$ip_address}:{$port}/testconnect/?token=".urlencode($token);
                $query_url_replaced = "https://{$ip_address_replaced}:{$port_replaced}/testconnect/?token=".urlencode($token_replaced);

                // Remove SSL verification or not
                $context = array("ssl" => array("verify_peer" => true, "verify_peer_name" => true));
                if ($no_ssl_verify) {
                    $context['ssl']['verify_peer'] = false;
                    $context['ssl']['verify_peer_name'] = false;
                }

                // All we want to do is test if we can hit this URL.
                $raw_json = file_get_contents($query_url_replaced, false, stream_context_create($context));
                if (empty($raw_json)) {
                    $errmsg[$errors++] = _("Unable to contact server at") . " {$query_url}.";
                } else {
                    $json = json_decode($raw_json, true);
                    if (!array_key_exists('value', $json)) {
                        $errmsg[$errors++] = _("Bad token for connection.");
                    }
                }

            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "");
            $token = grab_array_var($inargs, "token", "");
            $no_ssl_verify = grab_array_var($inargs, "no_ssl_verify", 1);
            $hostname = grab_array_var($inargs, 'hostname', gethostbyaddr($ip_address));
            $default_mem_units = grab_array_var($inargs, 'default_mem_units', 'Gi');

            $services_serial = grab_array_var($inargs, "services_serial", "");
            if ($services_serial) {
                $services = unserialize(base64_decode($services_serial));
            }

            // Load default services
            if (empty($services)) {
                $services = array();

                $services[] = array(
                    'check' => 1,
                    'name' => _('Application Error Logs'),
                    'log' => 'Application',
                    'severity' => 'ERROR'
                );

                $services[] = array(
                    'check' => 1,
                    'name' => _('System Error Logs'),
                    'log' => 'System',
                    'severity' => 'ERROR'
                );

                $services[] = array(
                    'check' => 1,
                    'name' => _('Security Error Logs'),
                    'log' => 'Security',
                    'severity' => 'ERROR'
                );
            }

            $output = '
<input type="hidden" name="ip_address" value="' . encode_form_val($ip_address) . '">
<input type="hidden" name="port" value="' . encode_form_val($port) . '">
<input type="hidden" name="token" value="' . encode_form_val($token) . '">
<input type="hidden" name="no_ssl_verify" value="' . intval($no_ssl_verify) . '">

<script type="text/javascript">
$(document).ready(function() {

    $(".add-box").click(function() {
        var id = $(this).data("id");

        // Update the id
        id = id+1;
        $(this).data("id", id);

        $(".services-list").append("<tr><td><input type=\'checkbox\' name=\'services[" + id + "][check]\' value=\'1\'></td><td><input type=\'text\' size=\'30\' name=\'services[" + id + "][name]\' value=\'\' class=\'form-control\'></td><td><input type=\'text\' size=\'15\' name=\'services[" + id + "][log]\' value=\'\' class=\'form-control\'></td><td class=\'form-inline\'><div style=\'margin-bottom: 5px;\'><div class=\'input-group\'><div class=\'input-group-addon\'>' . _('EventType') . '</div><select name=\'services[" + id + "][severity]\' class=\'form-control\'><option value=\'\'>'._('Any').'</option><option value=\'ERROR\'>'._('Error').'</option><option value=\'WARNING\'>'._('Warning').'</option><option value=\'INFORMATION\'>'._('Information').'</option><option value=\'AUDIT_FAILURE\'>'._('Audit Failure').'</option><option value=\'AUDIT_SUCCESS\'>'._('Audit Success').'</option><option value=\'UNKNOWN\'>'._('Unknown').'</option></select></div> <div class=\'input-group\'><div class=\'input-group-addon\'>' . _('EventID') . '</div><input type=\'text\' name=\'services[" + id + "][event_id]\' value=\'\' class=\'form-control\' size=\'5\'></div> <div class=\'input-group\'><div class=\'input-group-addon\'>' . _('SourceName') . '</div><input type=\'text\' name=\'services[" + id + "][application]\' value=\'\' class=\'form-control\' size=\'12\'></div> <div class=\'input-group\'><div class=\'input-group-addon\'>' . _('ComputerName') . '</div><input type=\'text\' name=\'services[" + id + "][computer_name]\' value=\'\' class=\'form-control\' size=\'12\'></div></div> <div><div class=\'input-group\'><div class=\'input-group-addon\'>' . _('EventCategory') . '</div><input type=\'text\' name=\'services[" + id + "][category]\' value=\'\' class=\'form-control\' size=\'15\'></div> <div class=\'input-group\'><div class=\'input-group-addon\'>' . _('Message') . '</div><input type=\'text\' name=\'services[" + id + "][message]\' value=\'\' placeholder=\''._('search or regex').'\' class=\'form-control\' size=\'20\'></div></div></td></tr>");
    });

});
</script>

<h5 class="ul">' . _("Connection Information") . '</h5>
<table class="table table-condensed table-no-border table-auto-width table-padded">
    <tr>
        <td class="vt"><label>' . _("Address") . ':</label></td>
        <td>
            <input type="text" size="20" value="' . encode_form_val($ip_address) . '" class="textfield form-control" disabled>
        </td>
    </tr>
    <tr>
        <td class="vt"><label>' . _("Host Name") . ':</label></td>
        <td>
            <input type="text" size="20" name="hostname" id="hostname" value="' . encode_form_val($hostname) . '" class="textfield form-control">
            <div class="subtext">' . _("The hostname you'd like to have associated with the new host") . '.</div>
        </td>
    </tr>
    <tr>
        <td class="vt"><label>' . _("Port") . ':</label></td>
        <td>
            <input type="text" size="5" value="' . encode_form_val($port) . '" class="textfield form-control" disabled>
        </td>
    </tr>
    <tr>
        <td><label>' . _("Token") . ':</label></td>
        <td>
            <input type="text" size="20" value="' . encode_form_val($token) . '" class="textfield form-control" disabled>
        </td>
    </tr>
</table>

<h5 class="ul">' . _('Event Log Checks') . '</h5>
<p>' . _('Specify what logs you are looking for and the filters that you want to apply to the logs. If any logs are found it will return a CRITICAL otherwise it will return an OK.') . '</p>
<table class="table table-condensed table-no-border table-auto-width">
    <thead>
        <tr>
            <td></td>
            <td>
                <label>' . _('Service Name') . ':</label>
            </td>
            <td>
                <label>' . _('Log') . ': <i class="fa fa-question-circle fa-14 tt-bind" title="' . _('The name of the windows event log. Examples: Application, Security, System, Setup') . '"></i></label>
            </td>
            <td>
                <label>' . _('Filters') . ': <i class="fa fa-question-circle fa-14 tt-bind" title="' . _('Optional. Add filters to only include specific logs.') . '"></i></label>
            </td>
        </tr>
    </thead>
    <tbody class="services-list">';

            foreach ($services as $sid => $svc) {
                $id = intval($sid);

                $output .= '<tr>';
                $output .= '<td><input type="checkbox" name="services[' . $id . '][check]" value="1" '.is_checked($svc['check'], 1).'></td>';
                $output .= '<td><input type="text" size="30" name="services[' . $id . '][name]" value="' . encode_form_val($svc['name']) . '" class="form-control"></td>';
                $output .= '<td><input type="text" size="15" name="services[' . $id . '][log]" value="' . encode_form_val($svc['log']) . '" class="form-control"></td>';
                $output .= '<td class="form-inline">
                <div style="margin-bottom: 5px;">
                    <div class="input-group">
                        <div class="input-group-addon">' . _('EventType') . '</div>
                        <select name="services[' . $id . '][severity]" class="form-control">
                            <option value="" '.is_selected($svc['severity'], '').'>'._('Any').'</option>
                            <option value="ERROR" '.is_selected($svc['severity'], 'ERROR').'>'._('Error').'</option>
                            <option value="WARNING" '.is_selected($svc['severity'], 'WARNING').'>'._('Warning').'</option>
                            <option value="INFORMATION" '.is_selected($svc['severity'], 'INFORMATION').'>'._('Information').'</option>
                            <option value="AUDIT_FAILURE" '.is_selected($svc['severity'], 'AUDIT_FAILURE').'>'._('Audit Failure').'</option>
                            <option value="AUDIT_SUCCESS" '.is_selected($svc['severity'], 'AUDIT_SUCCESS').'>'._('Audit Success').'</option>
                            <option value="UNKNOWN" '.is_selected($svc['severity'], 'UNKNOWN').'>'._('Unknown').'</option>
                        </select>
                    </div>
                    <div class="input-group">
                        <div class="input-group-addon">' . _('EventID') . '</div>
                        <input type="text" name="services[' . $id . '][event_id]" value="' . encode_form_val($svc['event_id']) . '" class="form-control" size="5">
                    </div>
                    <div class="input-group">
                        <div class="input-group-addon">' . _('SourceName') . '</div>
                        <input type="text" name="services[' . $id . '][application]" value="' . encode_form_val($svc['application']) . '" class="form-control" size="12">
                    </div>
                    <div class="input-group">
                        <div class="input-group-addon">' . _('ComputerName') . '</div>
                        <input type="text" name="services[' . $id . '][computer_name]" value="' . encode_form_val($svc['computer_name']) . '" class="form-control" size="12">
                    </div>
                </div>
                <div>
                    <div class="input-group">
                        <div class="input-group-addon">' . _('EventCategory') . '</div>
                        <input type="text" name="services[' . $id . '][category]" value="' . encode_form_val($svc['category']) . '" class="form-control" size="15">
                    </div>
                    <div class="input-group">
                        <div class="input-group-addon">' . _('Message') . '</div>
                        <input type="text" name="services[' . $id . '][message]" value="' . encode_form_val($svc['message']) . '" placeholder="'. _('search or regex') . '" class="form-control" size="20">
                    </div>
                </div>
                </td>';
                $output .= '</tr>';
            }

            $output .= '
    </tbody>
</table>

<div style="margin-bottom: 20px;">
    <a class="add-box" data-id="'.$id.'">'._('Add another check').'</a>
</div>
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:
            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');

            // Check for errors
            $errors = 0;
            $errmsg = array();
            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;

        case CONFIGWIZARD_MODE_GETSTAGE3OPTS:
            $outargs[CONFIGWIZARD_OVERRIDE_OPTIONS]["check_interval"] = 5;
            $outargs[CONFIGWIZARD_OVERRIDE_OPTIONS]["retry_interval"] = 1;
            $outargs[CONFIGWIZARD_OVERRIDE_OPTIONS]["max_check_attempts"] = 1;
            break;

        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');
            $services = grab_array_var($inargs, 'services', array());

            $output = '
            <input type="hidden" name="ip_address" value="' . encode_form_val($ip_address) . '" />
            <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '" />
            <input type="hidden" name="port" value="' . encode_form_val($port) . '" />
            <input type="hidden" name="token" value="' . encode_form_val($token) . '" />
            <input type="hidden" name="services_serial" value="' . base64_encode(serialize($services)) . '" />';

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:
            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:
            $output = '';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            // Get variables that were passed to us
            $ip_address = grab_array_var($inargs, 'ip_address');
            $hostname = grab_array_var($inargs, 'hostname');
            $port = grab_array_var($inargs, 'port');
            $token = grab_array_var($inargs, 'token');
            $services_serial = grab_array_var($inargs, "services_serial", "");
            $services = unserialize(base64_decode($services_serial));

            $objs = array();

            // Host
            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_ncpa_host",
                    "host_name" => $hostname,
                    "address" => $ip_address,
                    "icon_image" => "windowseventlog.png",
                    "statusmap_image" => "windowseventlog.png",
                    "_xiwizard" => $wizard_name
                );
            }

            // Common plugin opts
            $commonopts = "-t '$token' ";
            if ($port) {
                $commonopts .= "-P $port ";
            }

            // Services
            foreach ($services as $svc) {

                $pluginopts = $commonopts;

                // Verify we want to add the service
                if (trim($svc['name']) == "" || empty($svc['check'])) {
                    continue;
                }

                // Create query for plugin
                $query = 'name='.$svc['log'].',logged_after=30m';

                if (!empty($svc['severity'])) {
                    $query .= ',severity='.$svc['severity'];
                }
                if (!empty($svc['event_id'])) {
                    $query .= ',event_id='.$svc['event_id'];
                }
                if (!empty($svc['application'])) {
                    $query .= ',application='.$svc['application'];
                }
                if (!empty($svc['computer_name'])) {
                    $query .= ',computer_name='.$svc['computer_name'];
                }
                if (!empty($svc['category'])) {
                    $query .= ',category='.$svc['category'];
                }
                if (!empty($svc['message'])) {
                    $query .= ',message='.$svc['message'];
                }
                
                $pluginopts .= " -M 'logs' -q '".$query."' -c 0";

                $objs[] = array(
                    'type' => OBJECTTYPE_SERVICE,
                    'host_name' => $hostname,
                    'service_description' => $svc['name'],
                    'use' => 'xiwizard_ncpa_service',
                    'max_check_attempts' => 1,
                    'icon_image' => 'windowseventlog.png',
                    "statusmap_image" => "windowseventlog.png",
                    'check_command' => "check_xi_ncpa!" . $pluginopts,
                    '_xiwizard' => $wizard_name
                );
            }

            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;
            break;

        default:
            break;
    }

    return $output;
}
