<?php
//
// Linux Server Config Wizard
// Copyright (c) 2018-2022 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

s3_configwizard_init();

function s3_configwizard_init()
{
    $name = "s3";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "1.1.5",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor an Amazon S3 Bucket."),
        CONFIGWIZARD_DISPLAYTITLE => "Amazon S3",
        CONFIGWIZARD_FUNCTION => "s3_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "s3.png",
        CONFIGWIZARD_FILTER_GROUPS => array('amazon'),
        CONFIGWIZARD_REQUIRES_VERSION => 500
    );
    register_configwizard($name, $args);
}

/**
 * @return int
 *          0 = good to go
 *          1 = prereqs non-existent
 *          2 = may need to upgrade boto3
 */

function s3_configwizard_check_prereqs()
{
    // Plugin doesn't exist
    if (!file_exists("/usr/local/nagios/libexec/check_s3.py") && !is_dev_mode()) {
        return 4; // plugin non-existent
    }

    $preferred_version = '1.4.7';

    $found_pip = false;
    $pip_output = array();

    // See if boto3 is installed via pip and get the version number
    // $pip_command = "pip show boto3"; <-- THIS HAS LOW PERFORMANCE SOMETIMES, DO NOT USE
    $pip_command = 'python -c "import boto3"';
    exec($pip_command, $pip_output, $pip_return);

    // If neither yum nor pip returned anything, there is no need to continue
    if ($pip_return !== 0) {
        return 1; // prereqs non-existent
    }
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */

function s3_configwizard_func($mode = "", $inargs = null, &$outargs, &$result)
{

    $wizard_name = "s3";
    $local_url = get_base_url();

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {

        case CONFIGWIZARD_MODE_GETSTAGE1HTML:
            
            $check_prereqs = s3_configwizard_check_prereqs();

            // Remove any existing session variables if this is the first time on the page
            if (isset($_POST['backButton']) == false) {
                unset($_SESSION['bucketlist']);
                unset($_SESSION['checkboxlist']);
            }

            if ($check_prereqs == 1) {
                $output .= '<div class="message errorMessage" style="padding: 10px; margin-bottom: 20px;"><p><b>' . _('Error') . ':</b> ' . _('It looks like you are missing the python boto3 module on your Nagios XI server.') . '</p><p>' . _('To use this wizard you must install boto3 on your server. You can install it by running:') . '</p><pre>python -m pip install boto3</pre> OR <pre>pip install boto3</pre><p>' . _('If you do not see your buckets showing up on CentOS/RHEL 7, you may also need to install:') . '</p><pre>yum install python-futures</pre></div>';

            } elseif ($check_prereqs == 4) {
                $output .= '<div class="message errorMessage" style="padding: 10px; margin-bottom: 20px;"><p><b>' . _('Error') . ':</b> ' . _('It looks like you are missing the check_s3.py plugin on your Nagios XI server.</p></div>');
            } else {

            // Since this is the first step, we always want to reference the bucketlist stored in session
            $bucket_list = $_SESSION['bucketlist'];

            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");

            if ($credsfilepath == "") {
                $credsfilepath = "/usr/local/nagiosxi/etc/.aws/credentials";
            }

            if ($configfilepath == "") {
                $configfilepath = "/usr/local/nagiosxi/etc/.aws/config";
            }

            $output = '
            <h5 class="ul">'._('AWS Account Information').'</h5>
            <table class="table table-condensed table-no-border table-auto-width table-padded">
                <tr>
                    <td class="vt" style="width: 135px;">
                        <label>' . _('Access Key ID') . ':</label>
                    </td>
                    <td>
                        <input type="text" size="40" name="accesskeyid" id="accesskeyid" value="' . encode_form_val($accesskeyid) . '" class="textfield form-control credinput" ' . (checkbox_binary($staticcreds) ? "disabled" : "") . ' autocomplete="off">
                        <div class="subtext" id="accesskeyiderror" style="color: #c69; display: none;">' . _('Please enter an Access Key ID or specify a static credentials file') . '.</div>
                        <div class="subtext">' . _('The Access Key ID of the bucket to be monitored') . '.</div>
                    </td>
                </tr>
                <tr>
                    <td class="vt" style="width: 135px;">
                        <label>' . _('Secret Access Key') . ':</label>
                    </td>
                    <td>
                        <input type="text" size="40" name="accesskey" id="accesskey" value="' . encode_form_val($accesskey) . '" class="textfield form-control credinput" ' . (checkbox_binary($staticcreds) ? "disabled" : "") . ' autocomplete="off">
                        <div class="subtext" id="accesskeyerror" style="color: #c69; display: none;">' . _('Please enter a Secret Access Key or specify a static credentials file') . '.</div>
                        <div class="subtext">' . _('The Secret Access Key of the bucket to be monitored') . '.</div>
                    </td>
                </tr>
            </table>

            <table class="table table-condensed table-no-border table-auto-width">
                <tr>
                    <td style="width: 135px;"></td>
                    <td>
                        <input type="checkbox" class="checkbox" id="staticcreds" name="staticcreds" ' . is_checked(checkbox_binary($staticcreds), "1") . '>
                    </td>
                    <td>
                        <label class="normal" for="staticcreds">
                            <b>' . _('Static credentials file') . '</b><br>
                            ' . _('Use a statically defined AWS credentials file') . '.
                        </label>

                        <i class="credtooltip fa fa-question-circle fa-14" data-placement="right" data-content="If this box is checked, Nagios will read a local file containing your AWS credentials that follows Amazon\'s syntax.  Read more about credentials files <a target=\'' . '_blank' . '\' href=\'' . 'http://docs.aws.amazon.com/cli/latest/userguide/cli-config-files.html' . '\'>here</a>.">
                    </td>
                </tr>
            </table>

            <table class="table table-condensed table-no-border table-auto-width" id="credsfiletable">
                <tr>
                    <td class="vt" id="filepath" style="width: 135px;">
                        <label>' . _('File path') . ':</label>
                    </td>
                    <td>
                        <input type="text" size="40" name="credsfilepath" id="credsfilepath" value="' . encode_form_val($credsfilepath) . '" class="textfield form-control" disabled>
                        <div class="subtext">' . _('The absolute file path to the Amazon Credentials file (<b>Default:</b> /usr/local/nagiosxi/etc/.aws/credentials)') . '.</div>
                        <div class="subtext">' . _('<b>Note: Ensure the nagios group has read permissions for the entirety of the file path leading to the credentials file, including the file itself</b>') . '.</div>
                    </td>
                </tr>
            </table>

            <table class="table table-condensed table-no-border table-auto-width">
                <tr>
                    <td style="width: 135px;"></td>
                    <td>
                        <input type="checkbox" class="checkbox" id="staticconfig" name="staticconfig" ' . is_checked(checkbox_binary($staticconfig), "1") . '>
                    </td>
                    <td>
                        <label class="normal" for="staticconfig">
                            <b>' . _('Static configuration file') . '</b><br>
                            ' . _('Use a statically defined AWS configurations file') . '.
                        </label>
                        <i class="credtooltip fa fa-question-circle fa-14" data-placement="right" data-content="If this box is checked, Nagios will read a local file containing your AWS configuration that follows Amazon\'s syntax.  Read more about configuration files <a target=\'' . '_blank' . '\' href=\'' . 'http://docs.aws.amazon.com/cli/latest/userguide/cli-config-files.html' . '\'>here</a>.">
                    </td>
                </tr>
            </table>

            <table class="table table-condensed table-no-border table-auto-width" id="configfiletable">
                <tr>
                    <td class="vt" style="width: 135px;">
                        <label>' . _('File path') . ':</label>
                    </td>
                    <td>
                        <input type="text" size="40" name="configfilepath" id="configfilepath" value="' . encode_form_val($configfilepath) . '" class="textfield form-control" disabled>
                        <div class="subtext">' . _('The absolute file path to the Amazon Configuration file (<b>Default:</b> /usr/local/nagiosxi/etc/.aws/config)') . '.</div>
                        <div class="subtext">' . _('<b>Note: Ensure the nagios group has read permissions for the entirety of the file path leading to the config file, including the file itself</b>') . '.</div>
                    </td>
                </tr>
            </table>

            <h5 class="ul">'._('S3 Bucket Information').'</h5>
            <p>' . _('Select the buckets you would like to monitor, then enter the region name by the selected buckets.') . '</p>';

            if ($bucket_list == '') {
                $output .= '
                <div class="alert alert-info loader" id="bucket-notice" style="max-width: 500px; text-align: center; height: 117px;">
                    <div id="ajax-loader" class="sk-spinner sk-spinner-rotating-plane" style="background-color: #4D89F9; display: none; margin-top: 27px;">
                    </div>
                    <div id="info-placeholder">
                        <p class="center available-buckets" id="no-buckets" style="padding-top: 5px; padding-bottom: 15px;">We don\'t know which buckets are available yet! Fill out the AWS Account Information section, then click "Get Available Buckets".
                        </p>
                    </div>
                    <button type="button" id="get-buckets" class="btn btn-sm btn-default center available-buckets">Get Available Buckets</button>
               
                </div>
                <table class="table table-auto-width table-no-border" id="bucket-container">
                </table>';
            } else {
                $output .= '
                <table class="table table-auto-width table-no-border" id="bucket-container"><thead><tr><th class=\'vt\'><input type=\'checkbox\' id=\'select-all-buckets\' checked></th><th>' . _("Bucket Name") . '</th><th>' . _("Bucket Region") . '</th></tr></thead>';

                foreach ($bucket_list as $bucket_name => $bucket_info) {
                    $output .= '<tr><td class="vt"><input type="checkbox" class="bucket-list-checkbox" name="bucketlist[' . $bucket_name . '][checkbox]" style="margin-top: 2px;"' . (checkbox_binary($bucket_info["checkbox"]) ? "checked" : "") . '></td><td style="padding: 0px;"><input type="text" class="form-control" name="bucketlist[' . $bucket_name . '][bucket_name]" size="45" value="' . $bucket_name . '" readonly></td><td style="padding: 0 0 0 5px;"><input type="text" class="form-control" name="bucketlist[' . $bucket_name . '][region]" value="' . $bucket_info["region"] . '" readonly><input type="hidden" name="bucketlist[' . $bucket_name . '][metrics]" value="' . $bucket_info['metrics'] . '"></td></tr>';
                }

                $output .= '</table>';
            }
            
            $output .='
            <script type="text/javascript">

                $("#get-buckets").on("click", function() {

                    // Reset modal colors
                    $("#bucket-notice").css("background-color","#d9edf7");
                    $("#bucket-notice").css("border-color","#bce8f1");

                    // Check that we have all requisite data
                    var errors = 0;

                    var accesskeyid = $("#accesskeyid").val();
                    var accesskey = $("#accesskey").val();
                    var staticcreds = $("#staticcreds").prop("checked");

                    if ( ! accesskeyid && !staticcreds) {
                        $("#accesskeyid").css("border", "1px solid #c69");
                        $("#accesskeyiderror").show();
                        errors += 1;
                    } else {
                        $("#accesskeyiderror").hide();
                        $("#accesskeyid").css("border", "1px solid #ccc");
                    }

                    if ( ! accesskey && !staticcreds) {
                        $("#accesskey").css("border", "1px solid #c69");
                        $("#accesskeyerror").show();
                        errors += 1;
                    } else {
                        $("#accesskeyerror").hide();
                        $("#accesskey").css("border", "1px solid #ccc");
                    }

                    if (errors > 0) {
                        return false;
                    }

                    $("#get-buckets").hide();
                    $("#no-buckets").hide();
                    $("#info-placeholder").empty();
                    $("#ajax-loader").show();

                    var accesskeyid = $("#accesskeyid").val();
                    var accesskey = $("#accesskey").val();

                    // Base64 encode the strings so we can pass them as a query string
                    accesskeyid = btoa(accesskeyid);
                    accesskey = btoa(accesskey);

                    var formvalues = "mode=getbuckets";

                    if ($("#staticcreds").is(":checked")) {
                        var staticcreds = $("#staticcreds").val();
                        var credsfilepath = $("#credsfilepath").val();
                        formvalues += "&staticcreds=" + staticcreds;
                        formvalues += "&credsfilepath=" + credsfilepath;
                    }

                    formvalues += "&accesskeyid=" + accesskeyid;
                    formvalues += "&accesskey=" + accesskey;

                    full_url = base_url + "includes/configwizards/amazon_s3/s3.ajax.php?" + formvalues;

                    $.ajax({
                        url: full_url,
                        dataType: "json",
                        success: function(data) {

                            if ( ! data ) {
                                $("#ajax-loader").hide();
                                $("#bucket-notice").css("border", "1px solid #c69");
                                $("#bucket-notice").css("background-color", "#FF9999");
                                $("#info-placeholder").html(\'<p style="margin-bottom: 15px; padding-top: 5px; color: #333;">' . _("There were no buckets found. Verify your AWS credentials and try again") . '.</p>\');
                                $("#get-buckets").show();
                                return false;
                            }

                            $("#bucket-notice").hide();

                            $("#bucket-container").append("<thead><tr><th class=\'vt\'><input type=\'checkbox\' id=\'select-all-buckets\' checked></th><th>' . _("Bucket Name") . '</th><th>' . _("Bucket Region") . '</th></tr></thead>");

                            var counter = 0;

                            $.each(data, function( index, value ) {
                                $("#bucket-container").append(\'<tr><td class="vt"><input type="checkbox" class="bucket-list-checkbox" name="bucketlist[\' + index + \'][checkbox]" style="margin-top: 2px; margin-left: 7px;" checked></td><td><input type="text" class="form-control" name="bucketlist[\' + index + \'][bucket_name]" size="45" value="\' + index + \'" style="margin-bottom: 5px; margin-right: 5px;" readonly></td><td><input type="text" class="form-control" name="bucketlist[\' + index + \'][region]" value="\' + value + \'" style="margin-bottom: 5px;" readonly></td></tr>\');
                                counter++;
                            });
                        },
                        error: function(xmlhttprequest, textstatus, message) {
                            if (textstatus === "parsererror") {
                                $("#ajax-loader").hide();
                                $("#bucket-notice").css("border", "1px solid #c69");
                                $("#bucket-notice").css("background-color", "#FF9999");
                                $("#info-placeholder").html(\'<p style="margin-bottom: 15px; padding-top: 5px; color: #333;">There were no buckets found. Verify your AWS credentials and try again.</p>\');
                                $("#get-buckets").show();
                            }
                        }
                    });
                });

                $("#bucket-container").on("click", "thead > tr > th > input:checkbox", function() {
                    if ($(this).prop("checked") ==  true) {
                        $(".bucket-list-checkbox").prop("checked", true);
                    } else {
                        $(".bucket-list-checkbox").prop("checked", false);
                    }
                });

                $(function(){
                    $(".credtooltip").popover({ html: true });
                });

                document.addEventListener("DOMContentLoaded", function() {
                    var credtable = document.getElementById("credsfiletable");
                    var credinputs = document.getElementsByClassName("credinput");

                    document.querySelector("#staticcreds").addEventListener("change", function() {

                        for (var i = 0; i < credinputs.length; i++) {
                            if (credinputs[i].disabled === true) {
                                credinputs[i].disabled = false;
                            } else {
                                credinputs[i].disabled = true;
                            }
                        }
                    
                        if (document.getElementById("credsfilepath").disabled === true) {
                            document.getElementById("credsfilepath").disabled = false;
                        } else {
                            document.getElementById("credsfilepath").disabled = true;
                        }
                    });
                });

            $(function(){
                $(".configtooltip").popover({ html: true });
            });

            document.addEventListener("DOMContentLoaded", function() {
                var configtable = document.getElementById("configfiletable");
                var configinputs = document.getElementsByClassName("configinputs");

                document.querySelector("#staticconfig").addEventListener("change", function() {

                    for (var i = 0; i < configinputs.length; i++) {
                        if (configinputs[i].disabled === true) {
                            configinputs[i].disabled = false;
                        } else {
                            configinputs[i].disabled = true;
                        }
                    }

                    if (document.getElementById("configfilepath").disabled === true) {
                        document.getElementById("configfilepath").disabled = false;
                    } else {
                        document.getElementById("configfilepath").disabled = true;
                    }

                });
            });

            </script>';
            }
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            $bucket_list = grab_array_var($inargs, "bucketlist", "");

            // If we for some reason don't have a bucketlist, grab the session
            if ( $bucket_list == "" ) {
                $bucket_list = $_SESSION['bucketlist'];
            }

            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");
            $regionlist = grab_array_var($inargs, "regionlist", "");

            $staticcreds = grab_array_var($inargs, "staticcreds", "off");
            $staticconfig = grab_array_var($inargs, "staticconfig", "off");

            // Check for errors
            $errors = 0;
            $errmsg = array();
            $number_of_checked_buckets = 0;

            foreach ($bucket_list as $bucket_name => $bucket_info) {
                if ($bucket_info['checkbox'] == 'on') {
                    $number_of_checked_buckets++;
                }
            }

            if ($number_of_checked_buckets == 0) {
                $errmsg[$errors++] = _("You must select at least one bucket to continue");
            }

            if (empty($accesskeyid) and $staticcreds == "off") {
                $errmsg[$errors++] = _("Specify either an Access Key ID or a static credentials file");
            }

            if (empty($accesskey) and $staticcreds == "off") {
                $errmsg[$errors++] = _("Specify either a Secret Access Key or a static credentials file");
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            // Take each bucket name and get available metrics, store them in an array
            foreach($bucket_list as $bucket_name => $bucket_info) {

                if ( ! $bucket_info['checkbox'] ) {
                        // If the checkbox isn't on, skip it
                        continue;
                } else {

                    // Don't overwrite if we've been here before
                    if ( isset($bucket_info['metrics']) && ! empty($bucket_info['metrics']) ) {
                        continue;
                    }

                    // If static files are specified, use those
                    if ( $staticcreds !== 'off' && ! empty($credsfilepath)) {
                        $cmd = "/usr/local/nagios/libexec/check_s3.py --changemode 'getmetrics' --credfile '" . $credsfilepath . "' --bucketname '" . $bucket_name . "' --region '" . $bucket_info['region'] . "'";
                    } else {
                        // No static file defined
                        $cmd = "/usr/local/nagios/libexec/check_s3.py --changemode 'getmetrics' --accesskeyid '" . $accesskeyid . "' --secretaccesskey '" . $accesskey . "' --bucketname '" . $bucket_name . "' --region '" . $bucket_info['region'] . "'";
                    }

                    $cmd = escapeshellcmd($cmd);

                    $returnedmetrics = shell_exec($cmd);
                    $decodedmetrics = json_decode($returnedmetrics, true);

                    $merged_array = array();

                    foreach($decodedmetrics as $index => $metric) {
                        $merged_array[$index] = $metric;
                    }

                    $bucket_list[$bucket_name]['metrics'] = base64_encode(json_encode($merged_array));
                }
            }

            // Cache the bucketlist again after adding metrics
            $_SESSION['bucketlist'] = $bucket_list;

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us

            // $bucket_list = grab_array_var($inargs, "bucketlist", "");
            $bucket_list = $_SESSION['bucketlist'];
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");
            $regionlist = grab_array_var($inargs, "regionlist", "");
            $checkboxlist = grab_array_var($inargs, "checkboxlist", "");

            // Use to grab necessary information by condensed metric name
            $metrics_dictionary = array(
                'BucketSizeBytes' => array(
                    'name' => 'Bucket Size Bytes',
                    'description' => 'The amount of data in bytes stored in a bucket in the STANDARD storage class, INTELLIGENT_TIERING storage class, Standard - Infrequent Access (STANDARD_IA) storage class, OneZone - Infrequent Access (ONEZONE_IA), Reduced Redundancy Storage (RRS) class, or Glacier (GLACIER) storage class',
                    'units' => 'Bytes'
                ),
                'NumberOfObjects' => array(
                    'name' => 'Number of Objects',
                    'description' => 'The total number of objects (includes all objects including all their versions) stored in a bucket for all storage classes',
                    'units' => 'Objects'
                ),
                'AllRequests' => array(
                    'name' => 'All Requests',
                    'description' => 'The total number of HTTP requests made to an Amazon S3 bucket, regardless of type. If you\'re using a metrics configuration with a filter, then this metric only returns the HTTP requests made to the objects in the bucket that meet the filter\'s requirements',
                    'units' => 'Requests'
                ),
                'GetRequests' => array(
                    'name' => 'Get Requests',
                    'description' => 'The number of HTTP GET requests made for objects in an Amazon S3 bucket. This doesn\'t include list operations',
                    'units' => 'Requests'
                ),
                'PutRequests' => array(
                    'name' => 'Put Requests',
                    'description' => 'The number of HTTP PUT requests made for objects in an Amazon S3 bucket',
                    'units' => 'Requests'
                ),
                'DeleteRequests' => array(
                    'name' => 'Delete Requests',
                    'description' => 'The number of HTTP DELETE requests made for objects in an Amazon S3 bucket. This also includes Delete Multiple Objects requests. This metric shows the number of requests, not the number of objects deleted',
                    'units' => 'Requests'
                ),
                'HeadRequests' => array(
                    'name' => 'Head Requests',
                    'description' => 'The number of HTTP HEAD requests made to an Amazon S3 bucket',
                    'units' => 'Requests'
                ),
                'PostRequests' => array(
                    'name' => 'Post Requests',
                    'description' => 'The number of HTTP POST requests made to an Amazon S3 bucket',
                    'units' => 'Requests'
                ),
                'ListRequests' => array(
                    'name' => 'List Requests',
                    'description' => 'The number of HTTP requests that list the contents of a bucket',
                    'units' => 'Requests'
                ),
                'BytesDownloaded' => array(
                    'name' => 'Bytes Downloaded',
                    'description' => 'The number bytes downloaded for requests made to an Amazon S3 bucket, where the response includes a body',
                    'units' => 'Bytes'
                ),
                'BytesUploaded' => array(
                    'name' => 'Bytes Uploaded',
                    'description' => 'The number bytes uploaded that contain a request body, made to an Amazon S3 bucket',
                    'units' => 'Bytes'
                ),
                '4xxErrors' => array(
                    'name' => '4xx Errors',
                    'description' => 'The number of HTTP 4xx client error status code requests made to an Amazon S3 bucket with a value of either 0 or 1. The average statistic shows the error rate, and the sum statistic shows the count of that type of error, during each period',
                    'units' => 'Errors'
                ),
                '5xxErrors' => array(
                    'name' => '5xx Errors',
                    'description' => 'The number of HTTP 5xx server error status code requests made to an Amazon S3 bucket with a value of either 0 or 1. The average statistic shows the error rate, and the sum statistic shows the count of that type of error, during each period',
                    'units' => 'Errors'
                ),
                'FirstByteLatency' => array(
                    'name' => 'First Byte Latency',
                    'description' => 'The per-request time from the complete request being received by an Amazon S3 bucket to when the response starts to be returned',
                    'units' => 'Milliseconds'
                ),
                'TotalRequestLatency' => array(
                    'name' => 'Total Request Latency',
                    'description' => 'The elapsed per-request time from the first byte received to the last byte sent to an Amazon S3 bucket. This includes the time taken to receive the request body and send the response body, which is not included in First Byte Latency',
                    'units' => 'Milliseconds'
                ),
            );

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");
            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }
            if (!is_array($services)) {
                $services_default = array(
                    "ping" => 1,
                    "BucketSizeBytes" => 1,
                    "NumberOfObjects" => 1,
                    "AllRequests" => 1,
                    "GetRequests" => 1,
                    "PutRequests" => 1,
                    "DeleteRequests" => 1,
                    "HeadRequests" => 1,
                    "PostRequests" => 1,
                    "ListRequests" => 1,
                    "BytesDownloaded" => 1,
                    "BytesUploaded" => 1,
                    "4xxErrors" => 1,
                    "5xxErrors" => 1,
                    "FirstByteLatency" => 1,
                    "TotalRequestLatency" => 1

                );
                $services_default["servicestate"][0] = "on";
                $services_default["servicestate"][1] = "on";
                $services = grab_array_var($inargs, "services", $services_default);
            }

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");
            if ($serviceargs_serial != "") {
                $serviceargs_default = json_decode(base64_decode($serviceargs_serial), true);
            } else {
                $serviceargs_default = array(

                    "BucketSizeBytes_warning" => 1000000000,
                    "BucketSizeBytes_critical" => 1500000000,

                    "NumberOfObjects_warning" => 150,
                    "NumberOfObjects_critical" => 200,

                    "AllRequests_warning" => 1000,
                    "AllRequests_critical" => 1500,
                    
                    "GetRequests_warning" => 1000,
                    "GetRequests_critical" => 1500,
                    
                    "PutRequests_warning" => 1000,
                    "PutRequests_critical" => 1500,
                    
                    "DeleteRequests_warning" => 1000,
                    "DeleteRequests_critical" => 1500,
                    
                    "HeadRequests_warning" => 1000,
                    "HeadRequests_critical" => 1500,
                    
                    "PostRequests_warning" => 1000,
                    "PostRequests_critical" => 1500,
                    
                    "ListRequests_warning" => 1000,
                    "ListRequests_critical" => 1500,
                    
                    "BytesDownloaded_warning" => 500000000,
                    "BytesDownloaded_critical" => 750000000,
                    
                    "BytesUploaded_warning" => 500000000,
                    "BytesUploaded_critical" => 750000000,
                    
                    "4xxErrors_warning" => 25,
                    "4xxErrors_critical" => 50,
                    
                    "5xxErrors_warning" => 25,
                    "5xxErrors_critical" => 50,
                    
                    "FirstByteLatency_warning" => 150,
                    "FirstByteLatency_critical" => 300,
                    
                    "TotalRequestLatency_warning" => 150,
                    "TotalRequestLatency_critical" => 300
                );

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            if (!is_array($serviceargs)) {
                $serviceargs_default = array(

                    "BucketSizeBytes_warning" => 1000000000,
                    "BucketSizeBytes_critical" => 1500000000,

                    "NumberOfObjects_warning" => 150,
                    "NumberOfObjects_critical" => 200,

                    "AllRequests_warning" => 1000,
                    "AllRequests_critical" => 1500,
                    
                    "GetRequests_warning" => 1000,
                    "GetRequests_critical" => 1500,
                    
                    "PutRequests_warning" => 1000,
                    "PutRequests_critical" => 1500,
                    
                    "DeleteRequests_warning" => 1000,
                    "DeleteRequests_critical" => 1500,
                    
                    "HeadRequests_warning" => 1000,
                    "HeadRequests_critical" => 1500,
                    
                    "PostRequests_warning" => 1000,
                    "PostRequests_critical" => 1500,
                    
                    "ListRequests_warning" => 1000,
                    "ListRequests_critical" => 1500,
                    
                    "BytesDownloaded_warning" => 500000000,
                    "BytesDownloaded_critical" => 750000000,
                    
                    "BytesUploaded_warning" => 500000000,
                    "BytesUploaded_critical" => 750000000,
                    
                    "4xxErrors_warning" => 25,
                    "4xxErrors_critical" => 50,
                    
                    "5xxErrors_warning" => 25,
                    "5xxErrors_critical" => 50,
                    
                    "FirstByteLatency_warning" => 150,
                    "FirstByteLatency_critical" => 300,
                    
                    "TotalRequestLatency_warning" => 150,
                    "TotalRequestLatency_critical" => 300
                );

                $serviceargs = grab_array_var($inargs, "serviceargs", $serviceargs_default);
            }

            $main_css_url = get_base_url() . 'includes/configwizards/amazon_s3/css/main'.(get_theme() === 'xi5dark' ? '-dark' : '') . '.css';

            $output = '
            <link rel="stylesheet" href="'.$main_css_url.'"/>
            <input type="hidden" name="bucketlist" value="' . encode_form_val($bucket_list) . '">
            <input type="hidden" name="accesskeyid" value="' . encode_form_val($accesskeyid) . '">
            <input type="hidden" name="accesskey" value="' . encode_form_val($accesskey) . '">
            <input type="hidden" name="staticcreds" value="' . encode_form_val($staticcreds) . '">
            <input type="hidden" name="staticconfig" value="' . encode_form_val($staticconfig) . '">
            <input type="hidden" name="credsfilepath" value="' . encode_form_val($credsfilepath) . '">
            <input type="hidden" name="configfilepath" value="' . encode_form_val($configfilepath) . '">

            <h5 class="ul">' . _('S3 Bucket Metrics') . '</h5>';

            if ($staticcreds == "on") {
                $output .= '<input type="hidden" size="40" name="credsfilepath" id="credsfilepath" value="' . encode_form_val($credsfilepath) . '" class="textfield form-control credsinput" readonly>';
            } else {
                $output .=
                '<input type="hidden" name="accesskeyid" id="accesskeyid" value="' . encode_form_val($accesskeyid) . '" class="textfield form-control credsinput">
                 <input type="hidden" name="accesskey" id="accesskey" value="' . encode_form_val($accesskey) . '" class="textfield form-control credsinput">';
            }

            $output .= '
            <div id="advanced-wrapper" style="margin-top: 8px;">
                <div id="search-grid">
                    <div class="flex-center">
                        <p id="search-label">Filter Buckets:</p>
                        <input type="text" class="textfield form-control" id="bucket-search" size="35"/>
                    </div>
                    <div class="flex-center" id="search-subtext-wrapper">
                        <span class="advanced-subtext">' . _("Filter by Bucket Name or Region") . '</span>
                    </div>
                </div>
            </div>

            <div class="grid">
                <h6>' . _("Bucket Name") . '</h6>
                <h6>' . _("Region") . '</h6>
                <div class="flex-center">
                    <i class="fa fa-chevron-down" id="toggle-all-chevron"></i>
                </div>
            </div>';

            foreach($bucket_list as $bucket_name => $bucket_info) {

                // Don't let unchecked buckets pass through to step two
                if ( $bucket_info['checkbox'] != 'on' ) {
                    continue;
                }

                $output .= '
                <div class="bucket-wrapper" data-bucket-name="' . $bucket_name . '" data-region="' . $bucket_info["region"] . '">
                    <input type="hidden" name="bucketlist[' . $bucket_name . '][checkbox]" value="' . $bucket_info["checkbox"] . '">
                    <input type="hidden" name="bucketlist[' . $bucket_name . '][region]" value="' . $bucket_info["region"] . '">
                    <input type="hidden" name="bucketlist[' . $bucket_name . '][bucket_name]" value="' . $bucket_name . '">
                    <div class="grid bucket-header" data-bucket-name="' . $bucket_name . '">
                        <p>' . $bucket_name . '</p>
                        <p>' . $bucket_info["region"] . '</p>
                        <div class="flex-center">
                            <i class="fa fa-chevron-down metrics-chevron" data-bucket-name="' . $bucket_name . '"></i>
                        </div>
                    </div>
                    
                    <div class="metrics-grid-container" data-bucket-name="' . $bucket_name . '">
                        <div class="metrics-grid-table">

                            <!---------- Hostname ---------->
            
                            <div class="flex-center hostname-div">
                                <div class="input-group">
                                    <label class="input-group-addon tt-bind" title="" data-original-title="The name you want associated with this host" for="bucketlist[' . $bucket_name . '][hostname]">' . _("Hostname") . ' <i class="fa fa-question-circle" style="font-size: 12px; margin-left: 2px;"></i></label>
                                    <input name="bucketlist[' . $bucket_name . '][hostname]" value="' . $bucket_name . '" class="textarea form-control" type="text"/>
                                </div>
                            </div>

                            <!---------- Toggle All Checkbox ---------->

                            <div class="flex-center">
                                <input type="checkbox" class="checkbox toggle-checkbox" data-bucket-name="' . $bucket_name . '" checked="true">
                            </div>
                            <div class="toggle-all-wrapper">
                                <div id="toggle-all-label">
                                    <label class="select-cf-option">' . _("Toggle All") . '</label>
                                </div>
                            </div>';

                            // Loop over the metrics and create inputs for each
                            $metrics = json_decode(base64_decode($bucket_info['metrics']), true);
                            ksort($metrics);

                            foreach ($metrics as $metric_name => $metric_info) {

                                $formatted_metric_name = $metrics_dictionary[$metric_name]['name'];
                                $metric_description = $metrics_dictionary[$metric_name]['description'];
                                $metric_units = $metrics_dictionary[$metric_name]['units'];

                                $output .= '
                                <div class="flex-center">
                                    <input type="checkbox" class="checkbox ' . $bucket_name . '-checkbox" name="bucketlist[' . $bucket_name . '][metrics][' . $metric_name . '][checkbox]"  ' . is_checked(checkbox_binary($services["$metric_name"]), "1") . '>
                                </div>
                                <div class="flex-center">
                                    <div class="metric-grid">

                                        <!------- Metric Name ------->
                                        <div class="metric-name flex-center tt-bind" data-original-title="' . $metric_description . '.">
                                            <span>' . _("$formatted_metric_name") . ' <i class="fa fa-question-circle" style="font-size: 12px;"></i></span>
                                        </div>

                                        <!------- Warning and Critical Thresholds ------->
                                        <div class="input-group double-input">
                                            <!-- Warning -->
                                            <label class="input-group-addon no-border-bottom resize-label tt-bind" title="'._('Warning Threshold').'" style="border-right: 1px solid #ccc;"><img src="'.theme_image('error.png').'" ></label>
                                            <input data-bucket-name="' . $bucket_name . '" data-batch-edit-id="CPUCreditUsage_warning" name="bucketlist[' . $bucket_name . '][metrics][' . $metric_name . '][warning]" value="' . $serviceargs["$metric_name" . "_warning"] . '" class="textarea form-control batch-edit-target no-border-bottom" type="text"/>

                                            <!-- Critical -->
                                            <label class="input-group-addon no-border-bottom tt-bind resize-label" title="'._('Critical Threshold').'"><img src="'.theme_image('critical_small.png').'"></label>
                                            <input data-bucket-name="' . $bucket_name . '" data-batch-edit-id="CPUCreditUsage_critical" name="bucketlist[' . $bucket_name . '][metrics][' . $metric_name . '][critical]" value="' . $serviceargs["$metric_name" . "_critical"] . '" class="textarea form-control batch-edit-target no-border-bottom" type="text" />

                                            <!-- Units -->
                                            <label class="input-group-addon no-border-bottom" style="width: 100px;">' . _("$metric_units") . '</label>
                                        </div>

                                        <!------- Storage Type and Filter ID ------->
                                        <div class="input-group input-row-addon">
                                            <!-- Storage Type -->
                                            <label class="input-group-addon tt-bind" data-original-title="' . _("This dimension filters the data that you have stored in a bucket by the type of storage") . '.">' . _("Storage Type") . ' <i class="fa fa-question-circle" style="font-size: 12px;"></i></label>
                                            <input data-bucket-name="' . $bucket_name . '" data-batch-edit-id="' . $metric . '_warning" name="bucketlist[' . $bucket_name . '][metrics][' . $metric_name . '][storage_type]" value="' . $metric_info["StorageType"] . '" class="textarea form-control batch-edit-target resize-input" type="text" style="width: 178px;"/>

                                            <!-- Filter ID -->
                                            <label class="input-group-addon tt-bind" data-original-title="' . _("This dimension filters metrics configurations that you specify for request metrics on a bucket, for example, a prefix or a tag. You specify a filter id when you create a metrics configuration") . '." style="width: 112px;">' . _("Filter ID") . ' <i class="fa fa-question-circle" style="font-size: 12px;"></i></label>
                                            <input data-bucket-name="' . $bucket_name . '" data-batch-edit-id="' . $metric . '_critical" name="bucketlist[' . $bucket_name . '][metrics][' . $metric_name . '][filter_id]" value="' . $metric_info["FilterId"] . '" class="textarea form-control batch-edit-target" type="text" style="width: 280px;"/>
                                        </div>

                                    </div>
                                </div>';
                            }

                            $output .='
                    </div>
                </div>
            </div>';
            }

            if ($staticconfig == "on") {
                $output .= '<input type="hidden" name="configfilepath" id="configfilepath" value="' . encode_form_val($configfilepath) . '">';
            }

            $output .= '
            <div style="height: 20px;"></div>

            <script>
                function searchBuckets( searchString ) {
                    $(".bucket-wrapper").fadeOut(175);
                    
                    $(".bucket-wrapper").each(function() {
                        bucket_name = $(this).data("bucket-name");
                        region = $(this).data("region");
                        
                        bucket_name_result = bucket_name.indexOf(searchString);
                        region_result = region.indexOf(searchString);

                        if (bucket_name_result >= 0 || region_result >= 0) {
                            $(this).fadeIn(175);
                        }

                    });
                }


                $(document).ready(function() {
                    
                    $(".toggle-checkbox").on("click", function() {

                        var bucket_name = $(this).attr("data-bucket-name");
                        var checkbox_class = "." + bucket_name + "-checkbox";

                        if (this.checked) {
                            $(checkbox_class).prop("checked", true);
                        } else {
                            $(checkbox_class).prop("checked", false);
                        }
                        
                    });

                    var search;

                    // Instance Search
                    $("#bucket-search").on("keyup", function() {
                        clearTimeout(search);
                        searchString = $(this).val().toLowerCase();
                        search = setTimeout(\'searchBuckets(searchString)\', 400);
                    });

                    $("#toggle-all-chevron").on("click", function() {
                        if ( $(this).hasClass("expanded") ) {
                            $(this).removeClass("expanded");
                            $(this).css({ "transform" : "rotate(0deg)" });
                            $(".metrics-grid-container").each(function() {
                                $(this).css("height", "0px");
                                $(this).css("border-bottom", "0px");
                            });

                            $(".metrics-chevron").each(function() {
                                $(this).removeClass("expanded");
                                $(this).css({ "transform" : "rotate(0deg)" });
                            });

                            $(".bucket-header").each(function() {
                                $(this).removeClass("expanded");
                            });
                        } else {
                            $(this).addClass("expanded");
                            $(this).css({ "transform" : "rotate(180deg)" });
                            $(".metrics-grid-container").each(function() {
                                $(this).css("height", "auto");
                                $(this).css("border-bottom", "1px solid #ccc");
                            });

                            $(".metrics-chevron").each(function() {
                                $(this).addClass("expanded");
                                $(this).css({ "transform" : "rotate(180deg)" });
                            });

                            $(".bucket-header").each(function() {
                                $(this).addClass("expanded");
                            });
                        }
                    });

                    $(".bucket-header").on("click", function() {
                        
                        var bucketName = $(this).data("bucket-name");

                        if ( $(this).hasClass("expanded") ) {
                            $(this).removeClass("expanded");
                            $(\'.metrics-chevron[data-bucket-name="\' + bucketName + \'"]\').css({ "transform" : "rotate(0deg)" });
                            $(\'.metrics-grid-container[data-bucket-name="\' + bucketName + \'"]\').css("height", "0px");
                            $(\'.metrics-grid-container[data-bucket-name="\' + bucketName + \'"]\').css("border-bottom", "0px");
                        } else {
                            $(this).addClass("expanded");
                            $(\'.metrics-chevron[data-bucket-name="\' + bucketName + \'"]\').css({ "transform" : "rotate(180deg)" });
                            $(\'.metrics-grid-container[data-bucket-name="\' + bucketName + \'"]\').css("height", "auto");
                            $(\'.metrics-grid-container[data-bucket-name="\' + bucketName + \'"]\').css("border-bottom", "1px solid #ccc");
                        }

                    });

                });

                $("#toggle-all-instances").on("click", function() {
                    checked = $(this).prop("checked");
                    
                    if (checked) {
                        $(".instance-batch-checkbox").prop("checked", true);
                        $(".instance-batch-checkbox").trigger("change");
                    } else {
                        $(".instance-batch-checkbox").prop("checked", false);
                        $(".instance-batch-checkbox").trigger("change");
                    }
                });

                $(".instance-batch-checkbox").on("change", function() {
                    var checkbox = $(this);

                    if ( $(checkbox).prop("checked") ) {
                        $(checkbox).parent().parent().css("background-color", "#d9edf7");
                    } else {
                        $(checkbox).parent().parent().css("background-color", "white");
                    }
                });

                // Show batch edit modal
                $("#batch-edit-button").click(function() {
                    $("#batch-edit-modal-header").show();
                    $("#batch-edit-modal-content").show();
                });

                $("#hide-batch-edit").on("click", function() {
                    $("#batch-edit-modal-content").hide();
                    $("#batch-edit-modal-header").hide();
                });

                $("#apply-batch-edit-button").on("click", function () {

                    var checkedInstances = [];
                    var batchEdit = [];

                    // Get checked instances
                    $(".instance-batch-checkbox").each(function() {
                        if ( $(this).prop("checked") ) {
                            checkedInstances.push( $(this).data("instance-id") );
                        }
                    });

                    // Get all values
                    $(".batch-edit-input").each(function() {
                        // Check for checkboxes
                        if ( $(this).is(":checkbox") ) {
                            val = "checkbox";
                        } else {
                            val = $(this).val();
                        }

                        if (val) {
                            var editEntry = new Object;

                            var target = $(this).data("batch-edit-target");
                            if ( val === "checkbox") {
                                var value = $(this).prop("checked");
                            } else {
                                var value = $(this).val();
                            }

                            editEntry.target = target;
                            editEntry.value = value;

                            batchEdit.push(editEntry);
                        }
                    });

                    // Apply values to checked instances
                    for (index in batchEdit) {
                        targetId = batchEdit[index].target;
                        targetValue = batchEdit[index].value;

                        $(".batch-edit-target[data-batch-edit-id=" + targetId + "]").each(function() {
                            targetInstanceId = $(this).data("instance-id");
                            
                            if ( $.inArray(targetInstanceId, checkedInstances) >= 0 ) {
                                if ( $(this).is(":checkbox") ) {
                                    $(this).prop("checked", targetValue);
                                } else {
                                    $(this).val(targetValue);
                                }
                            }
                        });
                    }
                });
            </script>';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $bucket_list = grab_array_var($inargs, "bucketlist", "");
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");
            $region = grab_array_var($inargs, "region", "");

            // Cache our bucketlist
            $_SESSION['bucketlist'] = $bucket_list;

            // Check for errors
            $errors = 0;
            $errmsg = array();

            foreach ($hostnames as $host) {

                if (is_valid_host_name($host) == false) {
                    $errmsg[$errors++] = "[" . $host . "] Is an invalid host name.";
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $hostnames = grab_array_var($inargs, "hostnames");
            $bucket_list = grab_array_var($inargs, "bucketlist", "");
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");

            $output = '
            <input type="hidden" name="bucketlist" value="' . base64_encode(json_encode($bucket_list)) . '">
            <input type="hidden" name="accesskeyid" value="' . encode_form_val($accesskeyid) . '">
            <input type="hidden" name="accesskey" value="' . encode_form_val($accesskey) . '">
            <input type="hidden" name="region" value="' . encode_form_val($region) . '">
            <input type="hidden" name="staticcreds" value="' . encode_form_val($staticcreds) . '">
            <input type="hidden" name="staticconfig" value="' . encode_form_val($staticconfig) . '">
            <input type="hidden" name="credsfilepath" value="' . encode_form_val($credsfilepath) . '">
            <input type="hidden" name="configfilepath" value="' . encode_form_val($configfilepath) . '">';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            $check_interval = grab_array_var($inargs, "check_interval", "");

            // Check for errors
            $errors = 0;
            $errmsg = array();
            if ($check_interval < 5) {
                $errmsg[$errors++] = "Check interval cannot be less than five minutes. This is because AWS sends CloudWatch data every five minutes. Querying between the time the last data was was received and now - if less than five minutes - will result in an empty response from CloudWatch.";
            }
            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }
            break;

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            // Get variables that were passed to us
            $bucket_list = $_SESSION['bucketlist'];
            $accesskeyid = grab_array_var($inargs, "accesskeyid", "");
            $accesskey = grab_array_var($inargs, "accesskey", "");
            $staticcreds = grab_array_var($inargs, "staticcreds", "");
            $staticconfig = grab_array_var($inargs, "staticconfig", "");
            $credsfilepath = grab_array_var($inargs, "credsfilepath", "");
            $configfilepath = grab_array_var($inargs, "configfilepath", "");
            $check_interval = grab_array_var($inargs, "check_interval", "");

            $objs = array();

            // Append credentials file path if toggled, otherwise pass in bucketlist, accesskeyid and accesskey
            if ($staticcreds == "on") {
                $commonopts .= "--credfile '" . $credsfilepath . "' ";
            } else {
                $commonopts .= "--accesskeyid '" . $accesskeyid . "' --secretaccesskey '" . $accesskey . "' ";
            }

            // Append config file path if toggled, otherwise pass in region
            if ($staticconfig == "on") {
                $commonopts .= "--configfile '" . $configfilepath . "' ";
            }

            // Create non-existent hosts
            foreach ($bucket_list as $bucket_name => $bucket_info) {

                if ($bucket_info['checkbox'] != 'on') {
                    continue;
                }

                if (!host_exists($bucket_info["hostname"])) {
                    $objs[] = array(
                        "type" => OBJECTTYPE_HOST,
                        "use" => "xiwizard_linuxserver_host",
                        "host_name" => $bucket_info['hostname'],
                        "address" => $bucket_info['hostname'],
                        "icon_image" => "s3.png",
                        "statusmap_image" => "s3.png",
                        "_xiwizard" => $wizard_name,
                        "check_command" => "check_s3!" . "--changemode 'checkalive' --bucketname '" . $bucket_name . "' " . $commonopts
                    );
                }

            }

            $metric_name_dictionary = array(

                    "NumberOfObjects" => "Number of Objects",
                    "BucketSizeBytes" => "Bucket Size Bytes",
                    "AllRequests" => "All Requests",
                    "DeleteRequests" => "Delete Requests",
                    "GetRequests" => "Get Requests",
                    "HeadRequests" => "Head Requests",
                    "PutRequests" => "Put Requests",
                    "PostRequests" => "Post Requests",
                    "ListRequests" => "List Requests",
                    "BytesDownloaded" => "Bytes Downloaded",
                    "BytesUploaded" => "Bytes Uploaded",
                    "4xxErrors" => "4xx Errors",
                    "5xxErrors" => "5xx Errors",
                    "FirstByteLatency" => "First Byte Latency",
                    "TotalRequestLatency" => "Total Request Latency"

            );

            // See which services we should monitor
            foreach ($bucket_list as $bucket_name => $bucket_info) {

                $metrics = $bucket_info['metrics'];

                foreach ($metrics as $metric_name => $metric_info) {

                    if ($metric_info['checkbox'] != 'on') {
                        continue;
                    }

                    $dimensionopts = "";
                    
                    if ($metric_info['storage_type'] != "") {
                        $dimensionopts .= "--storagetype '" . $metric_info['storage_type'] . "' ";
                    }

                    if ($metric_info['filter_id'] != "") {
                        $dimensionopts .= "--filterid '" . $metric_info['filter_id'] . "' ";
                    }

                    if ($bucket_name != "") {
                        $dimensionopts .= "--bucketname '" . $bucket_name . "' ";
                    }

                    if ($bucket_info['region'] != "" && $statticconfig != "on") {
                        $dimensionopts .= "--region '" . $bucket_info['region'] . "' ";
                    }

                    switch ($metric_name) {

                        // Set check period of one day's worth of minutes for daily metrics 
                        case "BucketSizeBytes":
                        case "NumberOfObjects":

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $bucket_info['hostname'],
                                "service_description" => $bucket_info['hostname'] . " " . $metric_name_dictionary[$metric_name],
                                "use" => "xiwizard_linuxserver_ping_service",
                                "check_command" => "check_s3!" . "-P 1440" . " --metricname '" . $metric_name . "' " . $commonopts . "--warning '" . $metric_info['warning'] . "' --critical '" . $metric_info['critical'] . "' " . $dimensionopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;

                        case "Region":
                            break;

                        case "Hostname":
                            break;

                        default:
                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $bucket_info['hostname'],
                                "service_description" => $bucket_info['hostname'] . " " . $metric_name_dictionary[$metric_name],
                                "use" => "xiwizard_linuxserver_ping_service",
                                "check_command" => "check_s3!" . "-P " . $check_interval . " --metricname '" . $metric_name . "' " . $commonopts . "--warning '" . $metric_info['warning'] . "' --critical '" . $metric_info['critical'] . "' " . $dimensionopts,
                                "_xiwizard" => $wizard_name
                            );
                            break;
                    }
                }
            }

            // Return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


/**
 *
 * @return string
 */
