<?php
//
// SNMP Walk Config Wizard
// Copyright (c) 2008-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

snmpwalk_configwizard_init();

function snmpwalk_configwizard_init()
{
    $name = "snmpwalk";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "3.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Scan an SNMP-enabled device for elements to monitor."),
        CONFIGWIZARD_DISPLAYTITLE => _("SNMP Walk"),
        CONFIGWIZARD_FUNCTION => "snmpwalk_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "snmp.png",
        CONFIGWIZARD_FILTER_GROUPS => array('network'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function snmpwalk_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    global $cfg;

    $wizard_name = "snmpwalk";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "161");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "public");

            $snmpversion = grab_array_var($inargs, "snmpversion", "2c");
            $timeout = grab_array_var($inargs, "timeout", 15);
            $oid = grab_array_var($inargs, "oid", "");
            $maxresults = grab_array_var($inargs, "maxresults", 200);
            $mibs = grab_array_var($inargs, "mibs", array(''));

            $snmpopts = "";
            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if ($snmpopts_serial != "") {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            }

            if (!is_array($snmpopts)) {
                $snmpopts_default = array(
                    "v3_security_level" => "",
                    "v3_username" => "",
                    "v3_auth_password" => "",
                    "v3_privacy_password" => "",
                    "v3_auth_proto" => "",
                    "v3_priv_proto" => "",
                );

                $snmpopts = grab_array_var($inargs, "snmpopts", $snmpopts_default);
            }

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $address = nagiosccm_replace_user_macros($address);
            $port = grab_array_var($inargs, "port", "161");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "public");
            $snmpversion = grab_array_var($inargs, "snmpversion", "2c");
            $oid = grab_array_var($inargs, "oid", "");
            $maxresults = grab_array_var($inargs, "maxresults", 200);
            $timeout = grab_array_var($inargs, "timeout", 15);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false)
                $errmsg[$errors++] = "No address specified.";
            if (!is_numeric($port))
                $errmsg[$errors++] = "Port number must be numeric.";
            if (have_value($snmpcommunity) == false && $snmpversion != "3")
                $errmsg[$errors++] = "No SNMP community specified.";
            if (have_value($snmpversion) == false)
                $errmsg[$errors++] = "No SNMP version specified.";

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);

            $port = grab_array_var($inargs, "port", "161");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "public");
            $snmpversion = grab_array_var($inargs, "snmpversion", "2c");
            $oid = grab_array_var($inargs, "oid", "");
            $timeout = grab_array_var($inargs, "timeout", 15);
            $maxresults = grab_array_var($inargs, "maxresults", 200);
            $mibs = grab_array_var($inargs, "mibs", array(''));

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");

            if ($snmpopts_serial == "") {
                $snmpopts = grab_array_var($inargs, "snmpopts");
            } else {
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);
            }

            $services = "";
            $services_serial = grab_array_var($inargs, "services_serial", "");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            } else {
                $services = grab_array_var($inargs, "services", array());
            }

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            } else {
                $serviceargs = grab_array_var($inargs, "serviceargs", array());
            }

            // Only do the scan if we have no service args already
            if (empty($serviceargs)) {
                $resultfile = get_tmp_dir() . "/snmpwalk-" . escapeshellcmd($oid) . "-" . escapeshellcmd(nagiosccm_replace_user_macros($address));

                // snmp v3 stuff
                $cmdargs = "";
                if ($snmpversion == "3") {

                    $securitylevel = grab_array_var($snmpopts, "v3_security_level");
                    $username = grab_array_var($snmpopts, "v3_username");
                    $authproto = grab_array_var($snmpopts, "v3_auth_proto");
                    $authpassword = grab_array_var($snmpopts, "v3_auth_password");
                    $privacypassword = grab_array_var($snmpopts, "v3_privacy_password");
                    $privproto = grab_array_var($snmpopts, "v3_priv_proto");

                    if ($securitylevel != "")
                        $cmdargs .= " -l " . $securitylevel;
                    if ($username != "")
                        $cmdargs .= " -u " . nagiosccm_replace_user_macros($username);
                    if ($authproto != "")
                        $cmdargs .= " -a " . $authproto;
                    if ($authpassword != "")
                        $cmdargs .= " -A " . nagiosccm_replace_user_macros($authpassword);
                    if ($privacypassword != "")
                        $cmdargs .= " -X " . nagiosccm_replace_user_macros($privacypassword);
                    if ($privproto != "")
                        $cmdargs .= " -x " . $privproto;
                }

                if (empty($oid)) {
                    $useoid = "private";
                } else {
                    $useoid = escapeshellcmd($oid);
                }

                // Clean up empty mibs
                foreach ($mibs as $i => $m) {
                    if (empty($m)) {
                        unset($mibs[$i]);
                    }
                }

                // Replace user macros for fields we can use them in
                $cstring = nagiosccm_replace_user_macros($snmpcommunity);

                // Make command line and escape cmd args
                if ($snmpversion == "3") {
                    $cmdline = "/usr/bin/snmpwalk -Ob -m +ALL -v " . escapeshellcmd($snmpversion) . " " . escapeshellcmd($cmdargs) . " " . escapeshellcmd(nagiosccm_replace_user_macros($address)) . ":" . escapeshellcmd(nagiosccm_replace_user_macros($port)) . " " . $useoid . " > " . $resultfile . " 2>&1 & echo $!";
                } else {
                    $cmdline = "/usr/bin/snmpwalk -Ob -m +ALL -v " . escapeshellcmd($snmpversion) . " -c " . escapeshellcmd(nagiosccm_replace_user_macros($snmpcommunity)) . escapeshellcmd($cmdargs) . " " . escapeshellcmd(nagiosccm_replace_user_macros($address)) . ":" . escapeshellcmd(nagiosccm_replace_user_macros($port)) . " " . $useoid . " > " . $resultfile . " 2>&1 & echo $!";
                }

                // Run the command
                exec($cmdline, $op);
                $pid = (int)$op[0];

                // Wait until earlier of timeout or completion
                sleep(1);

                for ($x = 0; $x < $timeout; $x++) {
                    $out = array();

                    // See if process if still running...
                    exec("ps ax | grep $pid 2>&1", $out);
                    $check_pid = "";

                    foreach ($out as $row) {
                        $row_array = explode(" ", $row);
                        $check_pid = $row_array[0];

                        if ($pid == $check_pid) {
                            break;
                        }
                    }

                    // Process is gone - it must be done!
                    if ($check_pid != $pid)
                        break;

                    // Else process is still running
                    sleep(1);
                }

                // Read the results
                $fcontents = file_get_contents($resultfile);

                $rows = explode("\n", $fcontents);
                $x = 0;
                $hit_max = false;

                foreach ($rows as $row) {
                    // Get mib
                    $parts = explode("::", $row);
                    $mib = $parts[0];

                    if (!empty($mibs)) {
                        if (!empty($mib) && !in_array($mib, $mibs)) {
                            continue;
                        }
                    }

                    array_shift($parts);
                    $newrow = implode("::", $parts);

                    // Get oid
                    $parts = explode(" = ", $newrow);
                    $theoid = $parts[0];

                    array_shift($parts);
                    $newrow = implode(" = ", $parts);

                    // Get type
                    $parts = explode(":", $newrow);
                    $type = $parts[0];

                    array_shift($parts);
                    $newrow = implode(":", $parts);

                    // Get value
                    $val = $newrow;

                    // Skip if there is no value and no type
                    if (empty($val) && strpos($type, "No more variables left in this MIB View") !== false) {
                        continue;
                    }

                    // Make sure we have all the data
                    if ($mib == "" || $theoid == "" || $type == "") {
                        continue;
                    }

                    $x++;

                    if ($x > $maxresults) {
                        $hit_max = true;
                        break;
                    }

                    $serviceargs["oid"][] = array(
                        "oid" => $theoid,
                        "type" => $type,
                        "val" => $val,
                        "name" => "",
                        "label" => "",
                        "units" => "",
                        "matchtype" => "",
                        "warning" => "",
                        "critical" => "",
                        "string" => "",
                        "mib" => $mib,
                    );

                    $services["oid"][] = "";
                }
            }

            // Create mib list for display purposes only
            $miblist = '';

            if (!empty($mibs)) {
                $miblist = implode(', ', $mibs);
            }

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);

            $services = grab_array_var($inargs, "services", []);
            $serviceargs = grab_array_var($inargs, "serviceargs");

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if (!array_key_exists("oid", $services) || count($services["oid"]) == 0) {
                $errmsg[$errors++] = "You have not selected any OIDs to monitor.";
            } else {
                foreach ($services["oid"] as $index => $indexval) {
                    // get oid
                    $oid = $index;

                    // skip empty oids
                    if ($oid === "") {
                        continue;
                    }

                    // test match arguments
                    switch ($serviceargs["oid"][$index]["matchtype"]) {
                        case "numeric":
                            if ($serviceargs["oid"][$index]["warning"] == "")
                                $errmsg[$errors++] = "Invalid warning numeric range for OID " . intval($index) . ' (' . encode_form_val($serviceargs["oid"][$index]["oid"]) . ')';
                            if ($serviceargs["oid"][$index]["critical"] == "")
                                $errmsg[$errors++] = "Invalid critical numeric range for OID " . intval($index) . ' (' . encode_form_val($serviceargs["oid"][$index]["oid"]) . ')';
                            break;
                        case "string":
                            if ($serviceargs["oid"][$index]["string"] == "")
                                $errmsg[$errors++] = "Invalid string match for OID " . intval($index) . ' (' . encode_form_val($serviceargs["oid"][$index]["oid"]) . ')';
                            break;
                        default:
                            break;
                    }
                }
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = grab_array_var($inargs, "port");
            $hostname = grab_array_var($inargs, "hostname");
            $snmpcommunity = grab_array_var($inargs, "snmpcommunity");
            $snmpversion = grab_array_var($inargs, "snmpversion");
            $oid = grab_array_var($inargs, "oid", "");
            $timeout = grab_array_var($inargs, "timeout", 15);
            $mibs = grab_array_var($inargs, "mibs", array(''));
            $miblist = grab_array_var($inargs, "miblist");

            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");
            if ($snmpopts_serial == "")
                $snmpopts = grab_array_var($inargs, "snmpopts");
            else
                $snmpopts = json_decode(base64_decode($snmpopts_serial), true);

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="port" value="' . encode_form_val($port) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="snmpcommunity" value="' . encode_form_val($snmpcommunity) . '">
        <input type="hidden" name="snmpversion" value="' . encode_form_val($snmpversion) . '">
        <input type="hidden" name="oid" value="' . encode_form_val($oid) . '">
        <input type="hidden" name="miblist" value="' . encode_form_val($miblist) . '">
        <input type="hidden" name="timeout" value="' . encode_form_val($timeout) . '">
        <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
        <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">
        <input type="hidden" name="snmpopts_serial" value="' . base64_encode(json_encode($snmpopts)) . '">';

            foreach ($mibs as $m) {
                $output .= '<input type="hidden" name="mibs[]" value="'.encode_form_val($m).'">';
            }

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:

            $output = '

            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $hostaddress = $address;

            $port = grab_array_var($inargs, "port", "");

            $snmpcommunity = grab_array_var($inargs, "snmpcommunity", "");
            $snmpversion = grab_array_var($inargs, "snmpversion", "2c");

            $snmpopts_serial = grab_array_var($inargs, "snmpopts_serial", "");
            $snmpopts = json_decode(base64_decode($snmpopts_serial), true);

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["port"] = $port;
            $meta_arr["ip_address"] = $address;
            $meta_arr["snmpcommunity"] = $snmpcommunity;
            $meta_arr["snmpversion"] = $snmpversion;
            $meta_arr["services"] = $services;
            $meta_arr["serivceargs"] = $serviceargs;
            $meta_arr["snmpopts"] = $snmpopts;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_genericnetdevice_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "snmp.png",
                    "statusmap_image" => "snmp.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // see which services we should monitor
            foreach ($services as $svc => $svcstate) {

                //echo "PROCESSING: $svc -> $svcstate<BR>\n";

                switch ($svc) {

                    case "oid":

                        $enabledservices = $svcstate;

                        foreach ($enabledservices as $sid => $sstate) {

                            $oid = $serviceargs["oid"][$sid]["oid"];
                            $name = $serviceargs["oid"][$sid]["name"];
                            $label = $serviceargs["oid"][$sid]["label"];
                            $units = $serviceargs["oid"][$sid]["units"];
                            $matchtype = $serviceargs["oid"][$sid]["matchtype"];
                            $warning = $serviceargs["oid"][$sid]["warning"];
                            $critical = $serviceargs["oid"][$sid]["critical"];
                            $string = $serviceargs["oid"][$sid]["string"];
                            $mib = $serviceargs["oid"][$sid]["mib"];

                            // Strip whitespace from units
                            $units = str_replace(" ", "", $units);

                            $sdesc = $name;

                            $cmdargs = "";
                            // port
                            $cmdargs .= " -p " . intval($port);
                            // oid
                            if ($oid != "")
                                $cmdargs .= " -o " . escapeshellarg($oid);
                            // snmp community
                            if ($snmpcommunity != "" && $snmpversion != "3")
                                $cmdargs .= " -C " . escapeshellarg($snmpcommunity);
                            // snmp version
                            if ($snmpversion != "")
                                $cmdargs .= " -P " . $snmpversion;
                            // snmp v3 stuff
                            if ($snmpversion == "3") {

                                $securitylevel = grab_array_var($snmpopts, "v3_security_level");
                                $username = grab_array_var($snmpopts, "v3_username");
                                $authproto = grab_array_var($snmpopts, "v3_auth_proto");
                                $authpassword = grab_array_var($snmpopts, "v3_auth_password");
                                $privacypassword = grab_array_var($snmpopts, "v3_privacy_password");
                                $privproto = grab_array_var($snmpopts, "v3_priv_proto");

                                if ($securitylevel != "")
                                    $cmdargs .= " --seclevel=" . $securitylevel;
                                if ($username != "")
                                    $cmdargs .= " --secname=" . escapeshellarg($username);
                                if ($authproto != "")
                                    $cmdargs .= " --authproto=" . $authproto;
                                if ($authpassword != "")
                                    $cmdargs .= " --authpasswd=" . escapeshellarg($authpassword);
                                if ($privacypassword != "")
                                    $cmdargs .= " --privpasswd=" . escapeshellarg($privacypassword);
                                if ($privproto != "")
                                    $cmdargs .= " -x " . $privproto;
                            }
                            // label
                            if ($label != "")
                                $cmdargs .= " -l " . escapeshellarg($label);
                            // units
                            if ($units != "")
                                $cmdargs .= " -u " . escapeshellarg($units);
                            // mib
                            if ($mib != "")
                                $cmdargs .= " -m " . $mib;
                            // match type...
                            switch ($matchtype) {
                                case "numeric":
                                    if ($warning != "")
                                        $cmdargs .= " -w " . $warning;
                                    if ($critical != "")
                                        $cmdargs .= " -c " . $critical;
                                    break;
                                case "string":
                                    if ($string != "")
                                        $cmdargs .= " -r " . escapeshellarg($string);
                                    break;
                                default:
                                    break;
                            }

                            // make sure we have a service name
                            if ($sdesc == "")
                                $sdesc = $oid;

                            $objs[] = array(
                                "type" => OBJECTTYPE_SERVICE,
                                "host_name" => $hostname,
                                "service_description" => $sdesc,
                                "use" => "xiwizard_snmp_service",
                                "check_command" => "check_xi_service_snmp!" . $cmdargs,
                                "_xiwizard" => $wizard_name,
                            );
                        }
                        break;


                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}
