#!/bin/bash

# If no version argument provided, try to determine the version
if [ $# -lt 1 ]; then
    # Find the jar file
    WL_JAR=$(find / -name "weblogic.jar" 2>/dev/null | grep -v wlsagent)

    # If the jar is found, run command to get version info
    if [ "$WL_JAR" != "" ]; then
        WL_INFO=$(java -cp $WL_JAR weblogic.version | grep 'WebLogic Server [1-9][0-9]\.[0-9]\.[0-9]\.[0-9]\.[0-9]')
    fi

    # If we were unable to get version, ask user to enter it as an argument
    if [ "$WL_INFO" = "" ]; then
        echo ""
        echo "WARNING: Cannot determine weblogic version."
        echo "         Please enter version with command in this format:"
        echo "                 ./run.sh 12.3.4.5.6"
        echo ""
        exit 1
    fi

else
    # User entered a version argument. Use that instead.
    WL_INFO=$( echo "$1" | grep '[1-9][0-9]\.[0-9]\.[0-9]\.[0-9]\.[0-9]')

    # Confirm user entered a valid version number format
    if [ "$WL_INFO" = "" ]; then
        echo ""
        echo "WARNING: The argument you entered does not look like a version number."
        echo "         Please enter version with command in this format, including all five segments:"
        echo "                 ./run.sh 12.3.4.5.6"
        echo ""
        exit 1
    fi
fi

# Our version info is valid
# Check if version is 14.1.1.0.0 through 14.9.9.9.9
version14=$(echo $WL_INFO | grep '14\.[1-9]\.[1-9]\.[0-9]\.[0-9]')

# Check if version is 15.0.0.0.0 or above
version15plus=$(echo $WL_INFO | grep '1[5-9]\.[0-9]\.[0-9]\.[0-9]\.[0-9]')

# Check if version is 20.0.0.0.0 or above
version20plus=$(echo $WL_INFO | grep '[2-9][0-9]\.[0-9]\.[0-9]\.[0-9]\.[0-9]')



# We know which jar files to use. Begin install.
cd $(dirname "$0")

HOST="0.0.0.0"
PORT="9090"

CLASSPATH=".:wlsagent.jar"

# Add Jetty dependencies to the plugin classpath.
CLASSPATH="${CLASSPATH}:lib/servlet-api-2.5.jar:lib/jetty-servlet-7.6.2.v20120308.jar"


# Add WebLogic dependencies to the plugin classpath.
if [ "$version14" ] || [ "$version15plus" ] || [ "$version20plus" ]; then
    # For WebLogic versions 14.1.1.0.0 and above
    echo "Using newer jar files..."
    CLASSPATH="${CLASSPATH}:lib/wlthint3client.jar:lib/weblogic.jar:lib/com.oracle.weblogic.management.base.jar:lib/com.oracle.weblogic.work.jar"

else
    # For WebLogic versions less than 14.1.1.0.0
    # The 'wljmxclient.jar' must be loaded before the 'wlfullclient.jar'.
    echo "Using older jar files..."
    CLASSPATH="${CLASSPATH}:lib/wlclient.jar:lib/wljmxclient.jar:lib/wlfullclient.jar"
fi

# Launch WLS Agent
${JAVA_HOME}/bin/java -Xmx32m -cp ${CLASSPATH} net.wait4it.nagios.wlsagent.core.WLSAgent ${HOST} ${PORT} > /dev/null 2>&1 &

