<?php
//
// Oracle Tablespace Config Wizard
// Copyright (c) 2010-2024 Nagios Enterprises, LLC. All rights reserved.
//

include_once(dirname(__FILE__) . '/../configwizardhelper.inc.php');

oracletablespace_configwizard_init();

function oracletablespace_configwizard_init()
{
    $name = "oracletablespace";
    $args = array(
        CONFIGWIZARD_NAME => $name,
        CONFIGWIZARD_VERSION => "2.0.1",
        CONFIGWIZARD_TYPE => CONFIGWIZARD_TYPE_MONITORING,
        CONFIGWIZARD_DESCRIPTION => _("Monitor an Oracle Tablespace"),
        CONFIGWIZARD_DISPLAYTITLE => _("Oracle Tablespace"),
        CONFIGWIZARD_FUNCTION => "oracletablespace_configwizard_func",
        CONFIGWIZARD_PREVIEWIMAGE => "oracletablespace.png",
        CONFIGWIZARD_FILTER_GROUPS => array('database'),
        CONFIGWIZARD_REQUIRES_VERSION => 60100
    );
    register_configwizard($name, $args);
}

/**
 * @param string $mode
 * @param null   $inargs
 * @param        $outargs
 * @param        $result
 *
 * @return string
 */
function oracletablespace_configwizard_func($mode = "", $inargs = null, &$outargs = null, &$result = null)
{
    $wizard_name = "oracletablespace";

    // Initialize return code and output
    $result = 0;
    $output = "";

    // Initialize output args - pass back the same data we got
    $outargs[CONFIGWIZARD_PASSBACK_DATA] = $inargs;

    switch ($mode) {
        case CONFIGWIZARD_MODE_GETSTAGE1HTML:

            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "1521");
            $sid = grab_array_var($inargs, "sid", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "SYSTEM");

            # These will only be set by going back from step 2.
            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            // Sanity check for wizard requirements
            $sanity = '';

            if ((!file_exists('/usr/lib/oracle') && !file_exists('/usr/lib64/oracle')) || !file_exists('/usr/local/nagios/libexec/check_oracle_health') ) {
                ob_start();
?>
            <div class="alert alert-danger w-50 mt-2">
                <div class="align-items-center d-flex pb-3">
                    <i class="material-symbols-outlined md-400">warning</i>
                    <span class="fw-bold pe-2"><?= _('Warning') ?>:</span> <?=_("Oracle libraries do not appear to be installed.") ?>
                </div>
                <div class="card card-p">
                    <p class="text-start"><?= _('See the') ?> <a href='https://assets.nagios.com/downloads/nagiosxi/docs/Oracle_Plugin_Installation.pdf' title='Install Instructions' target='_blank'><?= _("Oracle Plugin Installation") ?></a> <?= _("instructions for monitoring Oracle.") ?></p>
                </div>
            </div>
<?php
                $sanity = ob_get_clean();
            }

            # Get the existing host/node configurations.
            # TODO: Include passwords/secrets?
            $nodes = get_configwizard_hosts($wizard_name);

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step1 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step1.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE1DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address", "");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");
            $sid = grab_array_var($inargs, "sid", "");

            $address = nagiosccm_replace_user_macros($address);
            $port = nagiosccm_replace_user_macros($port);
            $username = nagiosccm_replace_user_macros($username);
            $password = nagiosccm_replace_user_macros($password);

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (have_value($address) == false)
                $errmsg[$errors++] = _("No address specified.");
            if (have_value($port) == false)
                $errmsg[$errors++] = _("No port number specified.");
            if (have_value($username) == false)
                $errmsg[$errors++] = _("No username specified.");
            if (have_value($password) == false)
                $errmsg[$errors++] = _("No password specified.");
            if (have_value($database) == false)
                $errmsg[$errors++] = _("No tablespace specified.");

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;

        case CONFIGWIZARD_MODE_GETSTAGE2HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");
            $sid = grab_array_var($inargs, "sid", "");

            $ha = @gethostbyaddr($address);

            if ($ha == "") {
                $ha = $address;
            }

            $hostname = grab_array_var($inargs, "hostname", $ha);

            $services = grab_array_var($inargs, "services", array(
                "usage" => "on",
                "free" => "on",
                "remaining-time" => "on",
                "fragmentation" => "on",
                "io-balance" => "on",
                "can-allocate-next" => "on",
            ));

            $serviceargs = grab_array_var($inargs, "serviceargs", array(
                "usage_warning" => "90",
                "usage_critical" => "98",
                "free_warning" => "5:",
                "free_critical" => "2:",
                "remaining-time_warning" => "10",
                "remaining-time_critical" => "20",
                "fragmentation_warning" => "30:",
                "fragmentation_critical" => "20:",
                "io-balance_warning" => "50",
                "io-balance_critical" => "100",
                "can-allocate-next_warning" => "20",
                "can-allocate-next_critical" => "30",
            ));

            $select_options =[
                "Tablespace Usage" => "usage",
                "Tablespace Free Space" => "free",
                "Tablespace Remaining Time Until Full" => "remaining-time",
                "Tablespace Fragmentation" => "fragmentation",
                "Tablespace IO Balance" => "io-balance",
                "Can Allocate Next Extent" => "can-allocate-next",
            ];

            $service_tooltips = [
                "usage" => "Monitor amount of space used on the tablespace",
                "free" => "Monitor amount of space free on the tablespace",
                "remaining-time" => "Monitor indicators of how much time until tablespace is full",
                "fragmentation" => "Monitor indicators of how fragmented a tablespace is",
                "io-balance" => "Monitor indicators IO Balance",
                "can-allocate-next" => "Monitor segments of a tablespace which can allocate next extent",
            ];

            $services_serial = grab_array_var($inargs, "services_serial");

            if ($services_serial != "") {
                $services = json_decode(base64_decode($services_serial), true);
            }

            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial");

            if ($serviceargs_serial != "") {
                $serviceargs = json_decode(base64_decode($serviceargs_serial), true);
            }

            // // Build Multi select HTML
            // $select_options_html = '';
            // foreach ($select_options as $opt => $val) {
            //     $selected = '';
            //     if ($services[$val] == 'on') { $selected = ' selected'; }
            //     $select_options_html .= '<option value="'.$val.'"'.$selected.'>'.$opt.'</option>';
            // }

            // $select_html = '<select name="metrics[]" id="metrics" multiple  class="form form-control metrics-select multiselect form-select">';
            // $select_html .= $select_options_html;
            // $select_html .= '</select>';

            // Build Services Multi select HTML
            $service_list = '';
            $services_array = [];
            foreach ($select_options as $name => $service) {
                $name = encode_form_val($name);
                $safe_name = $service;
                $services_array[$name] = $service;
                $selected = '';
                if (isset($services[$service]) && $services[$service] == 'on'){ $selected = ' selected'; }
                $service_list .= "<option value='$safe_name' $selected>$name</option>";
            }

            $service_select_html = '<select name="services-select[]" id="services-select" multiple  class="form form-control metrics-select multiselect form-select">';
            $service_select_html .= $service_list;
            $service_select_html .= '</select>';

            ########################################################################################
            # Load the html
            # - The html needs to end up in the $output string, so use ob_start() and ob_get_clean()
            #   to load the PHP from the Step2 file into the $output string.
            ########################################################################################
            ob_start();
            include __DIR__.'/steps/step2.php';
            $output = ob_get_clean();

            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE2DATA:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $hostname = nagiosccm_replace_user_macros($hostname);
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $services = grab_array_var($inargs, "services", array());
            $serviceargs = grab_array_var($inargs, "serviceargs", array());

            // Check for errors
            $errors = 0;
            $errmsg = array();

            if (is_valid_host_name($hostname) == false) {
                $errmsg[$errors++] = "Invalid host name.";
            }

            if ($errors > 0) {
                $outargs[CONFIGWIZARD_ERROR_MESSAGES] = $errmsg;
                $result = 1;
            }

            break;


        case CONFIGWIZARD_MODE_GETSTAGE3HTML:

            // Get variables that were passed to us
            $address = grab_array_var($inargs, "ip_address");
            $hostname = grab_array_var($inargs, "hostname");
            $port = grab_array_var($inargs, "port", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");
            $sid = grab_array_var($inargs, "sid", "");

            $services = grab_array_var($inargs, "services");
            $serviceargs = grab_array_var($inargs, "serviceargs");

            $output = '

        <input type="hidden" name="ip_address" value="' . encode_form_val($address) . '">
        <input type="hidden" name="hostname" value="' . encode_form_val($hostname) . '">
        <input type="hidden" name="port" value="' . encode_form_val($port) . '">
        <input type="hidden" name="username" value="' . encode_form_val($username) . '">
        <input type="hidden" name="password" value="' . encode_form_val($password) . '">
        <input type="hidden" name="database" value="' . encode_form_val($database) . '">
        <input type="hidden" name="sid" value="' . encode_form_val($sid) . '">
        <input type="hidden" name="services_serial" value="' . base64_encode(json_encode($services)) . '">
        <input type="hidden" name="serviceargs_serial" value="' . base64_encode(json_encode($serviceargs)) . '">
';
            break;

        case CONFIGWIZARD_MODE_VALIDATESTAGE3DATA:

            break;

        case CONFIGWIZARD_MODE_GETFINALSTAGEHTML:


            $output = '
            ';
            break;

        case CONFIGWIZARD_MODE_GETOBJECTS:

            $hostname = grab_array_var($inargs, "hostname", "");
            $address = grab_array_var($inargs, "ip_address", "");
            $hostaddress = $address;
            $port = grab_array_var($inargs, "port", "");
            $sid = grab_array_var($inargs, "sid", "");
            $username = grab_array_var($inargs, "username", "");
            $password = grab_array_var($inargs, "password", "");
            $database = grab_array_var($inargs, "database", "");

            $services_serial = grab_array_var($inargs, "services_serial", "");
            $serviceargs_serial = grab_array_var($inargs, "serviceargs_serial", "");

            $services = json_decode(base64_decode($services_serial), true);
            $serviceargs = json_decode(base64_decode($serviceargs_serial), true);

            // save data for later use in re-entrance
            $meta_arr = array();
            $meta_arr["hostname"] = $hostname;
            $meta_arr["ip_address"] = $address;
            $meta_arr["port"] = $port;
            $meta_arr["username"] = $username;
            $meta_arr["sid"] = $sid;
            $meta_arr["password"] = $password;
            $meta_arr["database"] = $database;
            $meta_arr["services"] = $services;
            $meta_arr["serviceargs"] = $serviceargs;
            save_configwizard_object_meta($wizard_name, $hostname, "", $meta_arr);

            $objs = array();

            if (!host_exists($hostname)) {
                $objs[] = array(
                    "type" => OBJECTTYPE_HOST,
                    "use" => "xiwizard_oracletablespace_host",
                    "host_name" => $hostname,
                    "address" => $hostaddress,
                    "icon_image" => "oracle.png",
                    "statusmap_image" => "oracle.png",
                    "_xiwizard" => $wizard_name,
                );
            }

            // common plugin opts
            if (have_value($sid) == false) {
                $commonopts = "--connect '{$address}:{$port}' --username '{$username}' --password '{$password}' ";
            } else {
                $commonopts = "--connect '{$address}:{$port}/{$sid}' --username '{$username}' --password '{$password}' ";
            }

            foreach ($services as $svcvar => $svcval) {

                $pluginopts = "";
                $pluginopts .= $commonopts;

                switch ($svcvar) {

                    case "usage":

                        $pluginopts .= "--mode tablespace-usage  --warning " . $serviceargs["usage_warning"] . " --critical " . $serviceargs["usage_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Tablespace ".$database." Usage",
                            "use" => "xiwizard_oracletablespace_service",
                            "check_command" => "check_xi_oracletablespace!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "free":

                        $pluginopts .= "--mode tablespace-free  --warning " . $serviceargs["free_warning"] . " --critical " . $serviceargs["free_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Tablespace ".$database." Free Space",
                            "use" => "xiwizard_oracletablespace_service",
                            "check_command" => "check_xi_oracletablespace!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "remaining-time":

                        $pluginopts .= "--mode tablespace-remaining-time  --warning " . $serviceargs["remaining-time_warning"] . " --critical " . $serviceargs["remaining-time_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Tablespace ".$database." Remaining Time",
                            "use" => "xiwizard_oracletablespace_service",
                            "check_command" => "check_xi_oracletablespace!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "fragmentation":

                        $pluginopts .= "--mode tablespace-fragmentation  --warning " . $serviceargs["fragmentation_warning"] . " --critical " . $serviceargs["fragmentation_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Tablespace ".$database." Fragmentation",
                            "use" => "xiwizard_oracletablespace_service",
                            "check_command" => "check_xi_oracletablespace!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;


                    case "io-balance":

                        $pluginopts .= "--mode tablespace-io-balance  --warning " . $serviceargs["io-balance_warning"] . " --critical " . $serviceargs["io-balance_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Tablespace ".$database." IO Balance",
                            "use" => "xiwizard_oracletablespace_service",
                            "check_command" => "check_xi_oracletablespace!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    case "can-allocate-next":

                        $pluginopts .= "--mode tablespace-can-allocate-next  --warning " . $serviceargs["can-allocate-next_warning"] . " --critical " . $serviceargs["can-allocate-next_critical"] . "";

                        $objs[] = array(
                            "type" => OBJECTTYPE_SERVICE,
                            "host_name" => $hostname,
                            "service_description" => "Tablespace ".$database." Can Allocate Next",
                            "use" => "xiwizard_oracletablespace_service",
                            "check_command" => "check_xi_oracletablespace!" . $pluginopts,
                            "_xiwizard" => $wizard_name,
                        );
                        break;

                    default:
                        break;
                }
            }

            //echo "OBJECTS:<BR>";
            //print_r($objs);
            //exit();

            // return the object definitions to the wizard
            $outargs[CONFIGWIZARD_NAGIOS_OBJECTS] = $objs;

            break;

        default:
            break;
    }

    return $output;
}


?>
